package apiv2

import (
    "net/http"
    "testing"
    "net/http/httptest"
    "code.justin.tv/availability/goracle/goracleUser"
)

type goracleAuthTest struct {
    header        string
    response      string
    responseCode  int
    resultingUser *goracleUser.GoracleUser
}

var testCases = []goracleAuthTest{
    {
        "mmicch",
        "",
        http.StatusOK,
        &goracleUser.GoracleUser{
            "mmicch",
            "Michael Chang",
            102960821,
        },
    },
    {
        "",
        "401 - Must provide valid LDAP username as header in authUser",
        http.StatusUnauthorized,
        nil,
    },
    {
        "notALDAPUsername",
        "401 - Could not find given user in LDAP notALDAPUsername",
        http.StatusUnauthorized,
        nil,
    },
}

func TestS2SHandler(t *testing.T) {
    for _, test := range testCases {
        // Create a request to pass to our handler. We don't have any query parameters for now, so we'll
        // pass 'nil' as the third parameter.
        var user *goracleUser.GoracleUser
        basicHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
            user = goracleUser.GetUserFromContext(r.Context())
        })
        wrappedHandler := injectUserFromS2SCall(basicHandler)
        req, err := http.NewRequest("GET", "/test-user", nil)
        if test.header != "" {
            req.Header.Add("authUser", test.header)

        }
        if err != nil {
            t.Fatal(err)
        }

        // We create a ResponseRecorder (which satisfies http.ResponseWriter) to record the response.
        rr := httptest.NewRecorder()

        // Our handlers satisfy http.Handler, so we can call their ServeHTTP method
        // directly and pass in our Request and ResponseRecorder.
        wrappedHandler.ServeHTTP(rr, req)

        // Check the status code is what we expect.
        if status := rr.Code; status != test.responseCode {
            t.Errorf("handler returned wrong status code: got %v want %v",
                status, test.responseCode)
        }
        // Check the response body is what we expect.
        if rr.Body.String() != test.response {
            t.Errorf("handler returned unexpected body: got %v want %v",
                rr.Body.String(), test.response)
        }
        if user == nil || test.resultingUser == nil {
            if user != test.resultingUser {
                t.Errorf("handler was not able to parse correct user: got %v want %v",
                    user, test.resultingUser)
            }
            continue
        } else {
            if *user != *test.resultingUser {
                t.Errorf("handler was not able to parse correct user: got %v want %v",
                    user, test.resultingUser)
            }
        }
    }
}
