package apiv2

import (
	graphql "github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/catalog"
	"github.com/sirupsen/logrus"
)

type logRecordResolver struct {
	l *catalog.LogRecord
}

func (r *Resolver) Logs(
	args struct {
		ItemType *string
		ItemID   *graphql.ID
		Action   *string
		Author   *string
		Offset   *int32
		Limit    *int32
	}) ([]*logRecordResolver, error) {

	params := make(map[string]interface{})
	offset := -1
	limit := -1

	if args.ItemType != nil {
		params["item_type"] = *args.ItemType
	}
	if args.ItemID != nil {
		id, err := idStringToUint(*args.ItemID)
		if err != nil {
			logrus.Errorf("failed to convert log record id %s to string value due to: %s", *args.ItemID, err.Error())
			return nil, err
		}
		params["item_id"] = id
	}
	if args.Action != nil {
		params["action"] = *args.Action
	}
	if args.Author != nil {
		params["author"] = *args.Author
	}
	if args.Offset != nil {
		offset = int(*args.Offset)
	}
	if args.Limit != nil {
		limit = int(*args.Limit)
	}
	logs := catalog.GetCatalog().GetLogRecords(params, offset, limit)
	logRecordResolvers := []*logRecordResolver{}
	for _, log := range logs {
		logRecordResolvers = append(logRecordResolvers, &logRecordResolver{l: log})
	}
	return logRecordResolvers, nil
}

func resolveLogs(item_type string, item_id uint, offset, limit int) ([]*logRecordResolver, error) {
	params := map[string]interface{}{
		"item_type": item_type,
		"item_id":   item_id,
	}
	logs := catalog.GetCatalog().GetLogRecords(params, offset, limit)
	logRecordResolvers := []*logRecordResolver{}
	for _, log := range logs {
		logRecordResolvers = append(logRecordResolvers, &logRecordResolver{l: log})
	}
	return logRecordResolvers, nil
}

func (r *logRecordResolver) ID() graphql.ID {
	return idUintToString(r.l.ID)
}

func (r *logRecordResolver) Action() string {
	return r.l.Action
}

func (r *logRecordResolver) ItemType() string {
	return string(r.l.ItemType)
}

func (r *logRecordResolver) ItemLabel() string {
	return r.l.ItemLabel
}

func (r *logRecordResolver) ItemID() graphql.ID {
	return idUintToString(r.l.ItemID)
}

func (r *logRecordResolver) Before() string {
	return r.l.Before
}

func (r *logRecordResolver) After() string {
	return r.l.After
}

func (r *logRecordResolver) Author() string {
	return r.l.Author
}

func (r *logRecordResolver) Created() string {
	return r.l.CreatedAt.Format("2006-01-02T15:04:05.999999999-07:00")
}
