package apiv2

import (
	"context"

	graphql "github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/catalog"
)

// Queries
type metricResolver struct {
	m *catalog.Metric
}

func (r *Resolver) Metric(args struct{ ID graphql.ID }) (*metricResolver, error) {
	return resolveMetric(args.ID)
}

func (r *Resolver) Metrics(
	args struct {
		CalculationType *string
		ComponentID     *graphql.ID
		AttributeName   *string
		AttributeValue  *string
	},
) ([]*metricResolver, error) {
	ids, err := idsWithAttribute(catalog.LogTypeMetric, args.AttributeName, args.AttributeValue)
	if err != nil {
		return nil, err
	}
	// TODO: Support filters
	params := make(map[string]interface{})
	if args.CalculationType != nil {
		params["calculation_type"] = *args.CalculationType
	}
	if args.ComponentID != nil {
		id, err := idStringToUint(*args.ComponentID)
		if err != nil {
			return nil, err
		}
		params["component_id"] = id
	}
	metrics, err := catalog.GetCatalog().GetMetricsComplete(ids, params)
	if err != nil {
		return nil, err
	}

	metricResolvers := []*metricResolver{}
	for _, metric := range metrics {
		metricResolvers = append(metricResolvers, &metricResolver{m: metric})
	}
	return metricResolvers, nil
}

func resolveMetric(id graphql.ID) (*metricResolver, error) {
	u, err := idStringToUint(id)
	if err != nil {
		return nil, err
	}
	if u == 0 {
		return nil, nil
	}
	metric, err := catalog.GetCatalog().GetMetricByID(u)
	if err != nil {
		return nil, err
	}
	return &metricResolver{
		m: metric,
	}, nil
}

func (r *metricResolver) ID() graphql.ID {
	return idUintToString(r.m.ID)
}

func (r *metricResolver) Name() string {
	return r.m.Name
}

func (r *metricResolver) Label() string {
	return r.m.Label
}

func (r *metricResolver) Description() string {
	return r.m.Description
}

func (r *metricResolver) FeatureRollup() bool {
	return r.m.FeatureRollup
}

func (r *metricResolver) ComponentRollup() bool {
	return r.m.ComponentRollup
}

func (r *metricResolver) AutoGenerated() bool {
	return r.m.AutoGenerated
}

func (r *metricResolver) LatencyQuery() string {
	return string(r.m.LatencyQuery)
}

func (r *metricResolver) CalculationType() string {
	return string(r.m.CalculationType)
}

func (r *metricResolver) Threshold() float64 {
	return r.m.Threshold
}

func (r *metricResolver) LatencyObjective() float64 {
	return r.m.LatencyObjective
}

func (r *metricResolver) Component() (*componentResolver, error) {
	if r.m.ComponentID != 0 {
		id := idUintToString(r.m.ComponentID)
		return resolveComponent(id)
	}
	return nil, nil
}

func (r *metricResolver) ComponentID() *graphql.ID {
	if r.m.ComponentID != 0 {
		id := idUintToString(r.m.ComponentID)
		return &id
	}
	return nil
}

func (r *metricResolver) Queries() ([]*queryResolver, error) {
	// Check if we need to fetch the queries from the DB
	if r.m.Queries == nil {
		params := make(map[string]interface{})
		// There is an index on this field in the DB so this _should_ be a fast operation
		params["metric_id"] = r.m.ID
		queries, err := catalog.GetCatalog().GetQueriesComplete(nil, params)
		if err != nil {
			return nil, err
		}
		r.m.Queries = queries
	}
	// Now that we have the queries loaded, stuff them into resolvers
	queries := []*queryResolver{}
	for _, query := range r.m.Queries {
		queries = append(queries, &queryResolver{q: query})
	}
	return queries, nil
}

func (r *metricResolver) QueryIDs() ([]graphql.ID, error) {
	queryIDs := []graphql.ID{}
	for _, query := range r.m.Queries {
		queryIDs = append(queryIDs, idUintToString(query.ID))
	}
	return queryIDs, nil
}

func (r *metricResolver) Features() ([]*featureResolver, error) {
	features := []*featureResolver{}
	for _, feature := range r.m.Features {
		features = append(features, &featureResolver{f: feature})
	}
	return features, nil
}

func (r *metricResolver) FeatureIDs() ([]graphql.ID, error) {
	featureIDs := []graphql.ID{}
	for _, feature := range r.m.Features {
		featureIDs = append(featureIDs, idUintToString(feature.ID))
	}
	return featureIDs, nil
}

func (r *metricResolver) Attributes() []*attributeResolver {
	res, err := resolveAttributes(catalog.LogTypeMetric, r.m.ID)
	if err != nil {
		return nil
	}
	return res
}

// Mutations
type metricInput struct {
	ID               *graphql.ID
	Name             *string
	Label            *string
	Description      *string
	ComponentRollup  *bool
	Autogenerated    *bool
	Threshold        *float64
	ComponentID      *graphql.ID
	LatencyQuery     *string
	LatencyObjective *float64
	CalculationType  *string
	Queries          *[]queryInput
	Attributes       *[]*attributeInput `json:"-"`
}
type CreateMetricArgs struct {
	Metric *metricInput
}

func (r *Resolver) CreateMetric(ctx context.Context,
	args CreateMetricArgs,
) (*metricResolver, error) {

	// Create struct
	metric := &catalog.Metric{}

	if err := mergeMetricData(metric, args.Metric, r, ctx); err != nil {
		return nil, err
	}
	resolver, err := saveMetric(metric)
	if err != nil {
		return nil, err
	}
	r.saveAttributes(args.Metric.Attributes, resolver.ID(), catalog.LogTypeMetric, ctx)
	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, metric, ctx)

	return resolver, nil
}

type UpdateMetricArgs struct {
	ID     graphql.ID
	Metric *metricInput
}

func (r *Resolver) UpdateMetric(ctx context.Context,
	args UpdateMetricArgs,
) (*metricResolver, error) {
	// Get the existing metric
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	metric, err := catalog.GetCatalog().GetMetricByID(id)
	if err != nil {
		return nil, err
	}

	beforeLog := metric.LogInfo()

	// Update the given fields
	if err := mergeMetricData(metric, args.Metric, r, ctx); err != nil {
		return nil, err
	}

	resolver, err := saveMetric(metric)
	if err != nil {
		return nil, err
	}
	r.saveAttributes(args.Metric.Attributes, resolver.ID(), catalog.LogTypeMetric, ctx)
	afterLog := metric.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, beforeLog, afterLog, ctx)

	return resolver, nil

}

type DeleteMetricArgs struct {
	ID graphql.ID
}

func (r *Resolver) DeleteMetric(ctx context.Context,
	args DeleteMetricArgs,
) (*metricResolver, error) {

	// Get the existing metric
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	metric, err := catalog.GetCatalog().GetMetricByID(id)
	if err != nil {
		return nil, err
	}

	// Delete it
	err = catalog.GetCatalog().DeleteMetric(metric)
	if err != nil {
		return nil, err
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, metric, nil, ctx)

	return &metricResolver{m: metric}, nil
}

func saveMetric(metric *catalog.Metric) (*metricResolver, error) {
	err := catalog.GetCatalog().AddMetric(metric)
	if err != nil {
		return nil, err
	}
	return &metricResolver{m: metric}, nil
}

func mergeMetricData(dbComp *catalog.Metric, apiMet *metricInput, r *Resolver, ctx context.Context) error {
	if apiMet.Name != nil {
		dbComp.Name = *apiMet.Name
	}
	// optional, merge with default
	if apiMet.Label != nil {
		dbComp.Label = *apiMet.Label
	}
	if apiMet.Description != nil {
		dbComp.Description = *apiMet.Description
	}
	if apiMet.ComponentRollup != nil {
		dbComp.ComponentRollup = *apiMet.ComponentRollup
	}
	if apiMet.ComponentID != nil {
		compRes, err := resolveComponent(*apiMet.ComponentID)
		if err != nil {
			return err
		}
		if compRes != nil {
			dbComp.ComponentID = compRes.c.ID
			dbComp.Component = compRes.c
		} else {
			dbComp.ComponentID = 0
			dbComp.Component = nil
		}
	}
	if apiMet.Threshold != nil {
		dbComp.Threshold = *apiMet.Threshold
	}
	if apiMet.LatencyObjective != nil {
		dbComp.LatencyObjective = *apiMet.LatencyObjective
	}
	if apiMet.Queries != nil {
		queries := []*catalog.Query{}
		for _, query := range *apiMet.Queries {
			if query.ID == nil {
				args := CreateQueryArgs{Query: &query}
				res, err := r.CreateQuery(ctx, args)
				if err != nil {
					return err
				}
				queries = append(queries, res.q)
			} else {
				id, err := idStringToUint(*query.ID)
				if err != nil {
					return err
				}
				if id == 0 {
					args := CreateQueryArgs{Query: &query}
					res, err := r.CreateQuery(ctx, args)
					if err != nil {
						return err
					}
					queries = append(queries, res.q)
				} else {
					args := UpdateQueryArgs{ID: *query.ID, Query: &query}
					res, err := r.UpdateQuery(ctx, args)
					if err != nil {
						return err
					}
					queries = append(queries, res.q)
				}
			}
		}
		dbComp.Queries = queries
	}
	if apiMet.LatencyQuery != nil {
		dbComp.LatencyQuery = catalog.QueryType(*apiMet.LatencyQuery)
	}
	if apiMet.CalculationType != nil {
		dbComp.CalculationType = catalog.CalculationType(*apiMet.CalculationType)
	}
	return nil
}

func (r *metricResolver) Logs(
	args struct {
		Offset *int32
		Limit  *int32
	}) []*logRecordResolver {

	offset := -1
	limit := -1

	if args.Offset != nil {
		offset = int(*args.Offset)
	}
	if args.Limit != nil {
		limit = int(*args.Limit)
	}

	res, err := resolveLogs("metric", r.m.ID, offset, limit)
	if err != nil {
		return nil
	}
	return res
}
