package apiv2

import (
	"context"

	graphql "github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/catalog"
)

type queryResolver struct {
	q *catalog.Query
}

func (r *Resolver) Query(args struct{ ID graphql.ID }) (*queryResolver, error) {
	return resolveQuery(args.ID)
}

func (r *Resolver) Queries(args struct {
	AttributeName  *string
	AttributeValue *string
}) ([]*queryResolver, error) {
	ids, err := idsWithAttribute(catalog.LogTypeQuery, args.AttributeName, args.AttributeValue)
	if err != nil {
		return nil, err
	}
	queries, err := catalog.GetCatalog().GetQueriesByIDs(ids)
	if err != nil {
		return nil, err
	}
	queryResolvers := []*queryResolver{}
	for _, query := range queries {
		queryResolvers = append(queryResolvers, &queryResolver{q: query})
	}
	return queryResolvers, nil
}

func resolveQuery(id graphql.ID) (*queryResolver, error) {
	u, err := idStringToUint(id)
	if err != nil {
		return nil, err
	}
	query, err := catalog.GetCatalog().GetQueryByID(u)
	if err != nil {
		return nil, err
	}
	return &queryResolver{q: query}, nil
}

func (r *queryResolver) ID() graphql.ID {
	return idUintToString(r.q.ID)
}

func (r *queryResolver) QueryType() string {
	return string(r.q.QueryType)
}

func (r *queryResolver) AggregateType() string {
	return string(r.q.AggregateType)
}

func (r *queryResolver) Query() string {
	return r.q.Query
}

func (r *queryResolver) MetricID() *graphql.ID {
	if r.q.MetricID != 0 {
		id := idUintToString(r.q.MetricID)
		return &id
	}
	return nil
}

func (r *queryResolver) Metric() (*metricResolver, error) {
	if r.q.MetricID != 0 {
		return resolveMetric(*r.MetricID())
	}
	return nil, nil
}

func (r *queryResolver) Attributes() []*attributeResolver {
	res, err := resolveAttributes(catalog.LogTypeQuery, r.q.ID)
	if err != nil {
		return nil
	}
	return res
}

// Mutations
type queryInput struct {
	ID            *graphql.ID
	QueryType     *string
	AggregateType *string
	Query         *string
	MetricID      *graphql.ID
}
type CreateQueryArgs struct{ Query *queryInput }

func (r *Resolver) CreateQuery(ctx context.Context,
	args CreateQueryArgs,
) (*queryResolver, error) {

	// Create struct
	query := &catalog.Query{}

	if err := mergeQueryData(query, args.Query); err != nil {
		return nil, err
	}
	resolver, err := saveQuery(query)
	if err != nil {
		return nil, err
	}

	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, query, ctx)

	return resolver, nil
}

type UpdateQueryArgs struct {
	ID    graphql.ID
	Query *queryInput
}

func (r *Resolver) UpdateQuery(ctx context.Context,
	args UpdateQueryArgs,
) (*queryResolver, error) {
	// Get the existing query
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	query, err := catalog.GetCatalog().GetQueryByID(id)
	if err != nil {
		return nil, err
	}

	beforeLog := query.LogInfo()

	// Update the given fields
	if err := mergeQueryData(query, args.Query); err != nil {
		return nil, err
	}

	resolver, err := saveQuery(query)
	if err != nil {
		return nil, err
	}
	afterLog := query.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, beforeLog, afterLog, ctx)

	return resolver, nil

}

func (r *Resolver) DeleteQuery(ctx context.Context,
	args struct {
		ID graphql.ID
	},
) (*queryResolver, error) {

	// Get the existing query
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	query, err := catalog.GetCatalog().GetQueryByID(id)
	if err != nil {
		return nil, err
	}

	// Delete it
	err = catalog.GetCatalog().DeleteQuery(query)
	if err != nil {
		return nil, err
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, query, nil, ctx)

	return &queryResolver{q: query}, nil
}

func saveQuery(query *catalog.Query) (*queryResolver, error) {
	err := catalog.GetCatalog().AddQuery(query)
	if err != nil {
		return nil, err
	}
	return &queryResolver{q: query}, nil
}

func mergeQueryData(dbQuery *catalog.Query, apiQuery *queryInput) error {
	if apiQuery.QueryType != nil {
		dbQuery.QueryType = catalog.QueryType(*apiQuery.QueryType)
	}
	// optional, merge with default
	if apiQuery.AggregateType != nil {
		dbQuery.AggregateType = catalog.AggregateType(*apiQuery.AggregateType)
	}
	if apiQuery.Query != nil {
		dbQuery.Query = *apiQuery.Query
	}
	if apiQuery.MetricID != nil {
		metricRes, err := resolveMetric(*apiQuery.MetricID)
		if err != nil {
			return err
		}
		dbQuery.MetricID = metricRes.m.ID
	}
	return nil
}
