package apiv2

import (
	"code.justin.tv/availability/goracle/config"
	"encoding/csv"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"sort"
	"time"
)

// JSON obj structs
type AvailabilityReport struct {
	Date string
	Data []weeklyBuReport
}

type weeklyBuReport struct {
	ID             int
	Name           string
	ServiceReports []serviceReport
}

type serviceReport struct {
	SLA          float64
	ID           int32
	Availability float64
	Name         string
	TeamID       int
	TeamName     string
}

// TeamAvailabilityReportHandler creates CSV rollup of availability weekly reports.
func TeamAvailabilityReportHandler(w http.ResponseWriter, r *http.Request) {
	teamID := r.URL.Query().Get("teamID")

	fileName := "AvailabilityReport"
	if teamID != "" {
		fileName = fileName + "Team" + teamID
	}

	// These headers tell browser to download response to a csv file
	w.Header().Set("Content-Disposition", fmt.Sprintf("attachment; filename=%s.csv", fileName))
	w.Header().Set("Content-Type", r.Header.Get("Content-Type"))

	availabilityReports := getESKAvailReport(w)

	servicesToWeeklyReports, dates, err := extractInfo(availabilityReports, teamID)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	columns := []string{"service", "service_id", "latest_sla"}
	sort.Slice(dates, func(i, j int) bool { return dates[j].Before(dates[i]) })
	for _, d := range dates {
		columns = append(columns, d.Format("01/02/06"))
	}

	writeCSV(w, columns, servicesToWeeklyReports, dates)
}

// getESKAvailReport requests all availability reports from esk.
func getESKAvailReport(w http.ResponseWriter) []AvailabilityReport {
	availabilityReportURL := config.HMSEskEndpoint() + "reportitems/?reportID=availability"
	response, err := http.Get(availabilityReportURL)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return nil
	}
	defer response.Body.Close()
	body, readErr := ioutil.ReadAll(response.Body)
	if readErr != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return nil
	}
	var availabilityReports []AvailabilityReport
	jsonErr := json.Unmarshal(body, &availabilityReports)
	if jsonErr != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return nil
	}
	return availabilityReports
}

// extractInfo changes the availability reports into a service to weekly service reports map.
func extractInfo(availabilityReports []AvailabilityReport, teamID string) (map[int32]map[time.Time]serviceReport, []time.Time, error) {
	servicesToWeeklyReports := make(map[int32]map[time.Time]serviceReport)

	var dates []time.Time
	for _, weeklyReport := range availabilityReports {
		date, err := time.Parse(
			time.RFC3339,
			weeklyReport.Date)
		dates = append(dates, date)
		if err != nil {
			return nil, nil, err
		}
		for _, buRep := range weeklyReport.Data {
			for _, servRep := range buRep.ServiceReports {
				serviceKey := servRep.ID
				if servicesToWeeklyReports[serviceKey] == nil {
					servicesToWeeklyReports[serviceKey] = make(map[time.Time]serviceReport)
				}
				if teamID != "" && fmt.Sprintf("%d", servRep.TeamID) != teamID {
					continue
				}
				servicesToWeeklyReports[serviceKey][date] = servRep
			}
		}
	}
	return servicesToWeeklyReports, dates, nil
}

// writeCSV creates csv where each row is a service, it's latest sla, and all historical availability numbers for that service.
func writeCSV(w io.Writer, columns []string, servicesToReports map[int32]map[time.Time]serviceReport, dates []time.Time) {
	reportCSV := csv.NewWriter(w)
	reportCSV.Write(columns)
	for serviceID, serviceTimeseries := range servicesToReports {
		row := []string{}

		var serviceDates []time.Time
		for d := range serviceTimeseries {
			serviceDates = append(serviceDates, d)
		}
		if len(serviceDates) == 0 {
			continue
		}
		sort.Slice(serviceDates, func(i, j int) bool { return dates[j].Before(dates[i]) })

		latestDate := serviceDates[0]
		latestServiceReport := serviceTimeseries[latestDate]

		row = append(row, fmt.Sprintf(latestServiceReport.Name))
		row = append(row, fmt.Sprintf("%d", serviceID))
		row = append(row, fmt.Sprintf("%f", latestServiceReport.SLA))

		for _, d := range dates {
			r, ok := serviceTimeseries[d]
			if !ok {
				row = append(row, "")
			} else {
				row = append(row, fmt.Sprintf("%f", r.Availability))
			}
		}
		reportCSV.Write(row)
	}
	reportCSV.Flush()
}
