package apiv2

import (
	"context"
	"fmt"
	"strings"

	"github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/goracle/messaging"
	ldapLibrary "code.justin.tv/qe/twitchldap"
	"github.com/sirupsen/logrus"
	contacts "code.justin.tv/amzn/MyFultonContactsServiceLambdaTwirp"
	"code.justin.tv/availability/goracle/goracleUser"
	"errors"
	"encoding/json"
	"os"
	"time"
)

type serviceResolver struct {
	s *catalog.Service
}

func (r *Resolver) Service(args struct {
	ID graphql.ID
}) (*serviceResolver, error) {
	return resolveService(args.ID)
}

// ServicesForOwner takes in a employee UID and returns a list of services that that employee owns.
func (r *Resolver) ServicesForOwner(args struct {
	Uid *string
}) ([]*serviceResolver, error){
	if args.Uid == nil {
		return nil, fmt.Errorf("no owner was passed to ServicesForOwner function")
	}

	contactsService, err := getContactsServiceClient()
	if err != nil {
		logrus.Fatalf("unable to get contacts service client due to: %s", err.Error())
		return nil, err
	}

	anyLevelReports, err := contactsService.ListReports(context.Background(), &contacts.GetReportsRequest{UserId: *args.Uid, Depth: 10, IncludeManager: true})
	if err != nil {
		return nil, fmt.Errorf("failed to get reports from contacts service: %s", err.Error())
	}

	var serviceResolvers []*serviceResolver
	var empIds []interface{}
	for _, person := range anyLevelReports.Persons {
		empIds = append(empIds, person.EmployeeNumber)
	}
	services, err := catalog.GetCatalog().GetServices(map[string][]interface{}{"primary_owner_id": empIds})
	if err != nil {
		logrus.Errorf("Failed to query services for primary owner uid %d due to: %s", args.Uid, err.Error())
		return nil, err
	}
	for _, service := range services {
		serviceResolvers = append(serviceResolvers, &serviceResolver{s: service})
	}
	return serviceResolvers, nil
}

func (r *Resolver) Services(args struct {
	AttributeName  *string
	AttributeValue *string
}) ([]*serviceResolver, error) {
	ids, err := idsWithAttribute(catalog.LogTypeService, args.AttributeName, args.AttributeValue)
	if err != nil {
		logrus.Errorf("Failed to query IDs with name %s and value %s due to: %s", args.AttributeName, args.AttributeValue, err.Error())
		return nil, err
	}
	services, err := catalog.GetCatalog().GetServicesByIDs(ids)
	if err != nil {
		logrus.Errorf("Failed to query services with IDs %#v due to: %s", ids, err.Error())
		return nil, err
	}
	serviceResolvers := []*serviceResolver{}
	for _, service := range services {
		serviceResolvers = append(serviceResolvers, &serviceResolver{s: service})
	}
	return serviceResolvers, nil
}

func resolveService(id graphql.ID) (*serviceResolver, error) {
	u, err := idStringToUint(id)
	if err != nil {
		return nil, err
	}
	if u == 0 {
		return nil, nil
	}
	service, err := catalog.GetCatalog().GetServiceByID(u)
	if err != nil {
		return nil, err
	}
	return &serviceResolver{s: service}, nil
}

func (r *serviceResolver) ID() graphql.ID {
	return idUintToString(r.s.ID)
}

func (r *serviceResolver) Name() string {
	return r.s.Name
}

func (r *serviceResolver) Description() string {
	return r.s.Description
}

func (r *serviceResolver) AvailabilityObjective() float64 {
	return r.s.AvailabilityObjective
}

func (r *serviceResolver) Type() string {
	return r.s.ServiceType.Label
}

func (r *serviceResolver) Environment() string {
	return r.s.Environment
}

func (r *serviceResolver) Known() bool {
	return r.s.Known
}

func (r *serviceResolver) PrimaryOwnerID() *graphql.ID {
	if r.s.PrimaryOwnerID != 0 {
		i := idUintToString(r.s.PrimaryOwnerID)
		return &i
	}
	return nil
}

func (r *serviceResolver) PrimaryOwner() (*userResolver, error) {
	if r.s.PrimaryOwnerID != 0 {
		return resolveUser(idUintToString(r.s.PrimaryOwnerID))
	}
	return nil, nil
}

func (r *serviceResolver) Created() string {
	return r.s.CreatedAt.Format("2006-01-02T15:04:05.999999999-07:00")
}

func (r *serviceResolver) Updated() string {
	return r.s.UpdatedAt.Format("2006-01-02T15:04:05.999999999-07:00")
}

func (r *serviceResolver) Components() []*componentResolver {
	components := []*componentResolver{}
	for _, comp := range r.s.Components {
		components = append(components, &componentResolver{c: comp})
	}
	return components
}

func (r *serviceResolver) ComponentIDs() []graphql.ID {
	components := []graphql.ID{}
	for _, comp := range r.s.Components {
		components = append(components, idUintToString(comp.ID))
	}
	return components
}

func (r *serviceResolver) Attributes() []*attributeResolver {
	res, err := resolveAttributes(catalog.LogTypeService, r.s.ID)
	if err != nil {
		logrus.Errorf("failed to resolve attributes for service with ID %d", r.s.ID)
		return nil
	}
	return res
}
func (r *serviceResolver) ServiceDownstreams() []*serviceDependencyResolver {
	res, err := resolveServiceDependencies(&r.s.ID, nil)
	if err != nil {
		logrus.Errorf("failed to resolve service dependencies for service with ID %d due to: %s", r.s.ID, err.Error())
		return nil
	}
	return res
}
func (r *serviceResolver) ServiceUpstreams() []*serviceDependencyResolver {
	res, err := resolveServiceDependencies(nil, &r.s.ID)
	if err != nil {
		logrus.Errorf("failed to resolve service dependencies for service with ID %d due to: %s", r.s.ID, err.Error())
		return nil
	}
	return res
}

func (r *serviceResolver) ServiceAudits() []*serviceAuditResolver {
	res, err := resolveServiceAudits(r.s.ID, false)
	if err != nil {
		logrus.Errorf("failed to resolve audits for service with ID %d due to: %s", r.s.ID, err.Error())
		return nil
	}
	return res
}

func (r *serviceResolver) LatestServiceAudits() []*serviceAuditResolver {
	res, err := resolveServiceAudits(r.s.ID, true)
	if err != nil {
		logrus.Errorf("failed to resolve latest service dependencies for service with ID %d due to: %s", r.s.ID, err.Error())
		return nil
	}
	return res
}

func (r *serviceResolver) PagerDuty() string {
	return r.s.PagerDuty
}

// backwards compat, remove later
// TODO(mmicch) remove this after we are fully on slack channels
func (r *serviceResolver) Slack() string {
	if messaging.IsSlackChannelName(r.s.Slack) {
		return r.s.Slack
	}
	channel, err := resolveSlackChannel(graphql.ID(r.s.Slack))
	if err != nil {
		logrus.Errorf("failed to query slack channel name for service with ID %d and slack channel %s due to: %s", r.s.ID, r.s.Slack, err.Error())
		return ""
	}
	return channel.Name()
}

func (r *serviceResolver) SlackChannelId() *graphql.ID {
	if r.s.Slack == "" {
		return nil
	}
	id := graphql.ID(r.s.Slack)
	if messaging.IsSlackChannelName(r.s.Slack) {
		channel, err := resolveSlackChannel(id)
		if err != nil {
			logrus.Errorf("failed to query slack channel ID for service with ID %d and slack channel %s due to: %s", r.s.ID, r.s.Slack, err.Error())
			return nil
		}
		id = graphql.ID(channel.ID())
	}
	return &id
}

func (r *serviceResolver) SlackChannel() (*slackChannelResolver, error) {
	if r.s.Slack != "" {
		channel, err := resolveSlackChannel(graphql.ID(r.s.Slack))
		if err != nil {
			logrus.Errorf("failed to query slack channel for service with ID %d and slack channel %s due to: %s", r.s.ID, r.s.Slack, err.Error())
            return nil, nil
		}
		return channel, nil
	}
	return nil, nil
}

// State() returns state name
func (r *serviceResolver) State() string {
	return r.s.State.Name
}

// Mutations
type serviceInput struct {
	ID                    *graphql.ID
	Name                  *string
	Description           *string
	PrimaryOwnerID        *graphql.ID
	PrimaryOwnerUID       *string
	AvailabilityObjective *float64
	Environment           *string
	Known                 *bool
	Type                  *string
	ComponentIDs          *[]graphql.ID
	Components            *[]componentInput
	Attributes            *[]*attributeInput `json:"-"`
	ServiceUpstreams      *[]*serviceDependencyInput
	ServiceDownstreams    *[]*serviceDependencyInput
	PagerDuty             *string
	Slack                 *string
	SlackChannelID        *graphql.ID
	State                 *string
}

func (s serviceInput) String() string {
	return fmt.Sprintf("{ID: %d, Name: %s, PrimaryOwnerUID: %s, Slack: %s}", s.ID, safeString(s.Name), safeString(s.PrimaryOwnerUID), safeString(s.Slack))
}

// create a service audit validating the primary ownership if primary owner is creating this service, or assigning to self
func autoValidateIfRequired(ctx context.Context, service *catalog.Service) {
	// get current authenticated user
	currentUser := goracleUser.GetUserFromContext(ctx)
	if currentUser == nil {
		logrus.Errorf("Auto-Validation Error: Failed to retrieve current user from guardian")
		return
	}
	// get primary owner's name
	ldapClient, err := ldapLibrary.NewClient()
	if err != nil {
		logrus.Errorf("Auto-Validation Error: Failed to initialize ldap client: %s", err.Error())
		return
	}
	primaryOwner, err := ldapClient.GetUserInfo(uint32(service.PrimaryOwnerID))
	if err != nil || primaryOwner == nil {
		logrus.Errorf("Auto-Validation Error: Failed to get user info from ldap client for primaryOwnerID %s: %s", service.PrimaryOwnerID, err.Error())
		return
	}
	// add primary owner audit
	if currentUser.UID == primaryOwner.UID {
		// Create validation struct
		serviceAudit := &catalog.ServiceAudit{Auditor: currentUser.UID, ServiceID: service.ID, Action: "validated", AuditType: "owner", AuditValue: primaryOwner.CN}
		serviceAudit.AuditTime = time.Now()
		// Add service audit
		err = catalog.GetCatalog().AddServiceAudit(serviceAudit)
		if err != nil {
			logrus.Errorf("Auto-Validation Error: Failed to add service audit (primary owner) for serviceID: %s", service.ID)
		}
	}
	// add service state audit. always automatically validate the value (set by user) for creation and update operations
	serviceAudit := &catalog.ServiceAudit{Auditor: currentUser.UID, ServiceID: service.ID, Action: "validated", AuditType: "state", AuditValue: service.State.Name}
	serviceAudit.AuditTime = time.Now()
	err = catalog.GetCatalog().AddServiceAudit(serviceAudit)
	if err != nil {
		logrus.Errorf("Auto-Validation Error: Failed to add service audit (state) for serviceID: %d", service.ID)
	}
}

func (r *Resolver) CreateService(ctx context.Context,
	args struct {
		Service *serviceInput
	},
) (*serviceResolver, error) {
	service := catalog.DefaultService()
	if err := mergeServiceData(service, args.Service, ctx, r); err != nil {
		logrus.Errorf("Failed to merge data while creating service %s due to: %s", args.Service, err.Error())
		return nil, err
	}
	// create a new service
	resolver, err := saveService(service)
	if err != nil {
		logrus.Errorf("Failed to persist service %s due to: %s", args.Service, err.Error())
		return nil, err
	}
	// update upstream & downstream dependencies of the service
	// this updates service attributes but does not save the service
	if err := postSaveUpdates(service, args.Service, ctx, r); err != nil {
		logrus.Errorf("Failed post-save procedure while creating service %s due to: %s", args.Service, err.Error())
		return nil, err
	}
	// save service with updated attributes
	resolver, err = saveService(service)
	if err != nil {
		logrus.Errorf("Failed to persist updated attributes on service %s due to: %s", args.Service, err.Error())
		return nil, err
	}
	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, service, ctx)

	// send notification to Primary Owner
	primaryOwner, err := getLDAPUserFromID(uint32(service.PrimaryOwnerID))
	if err := notifyServiceOwner(service, primaryOwner.UID, primaryOwner.Mail, primaryOwner.CN, "", "", "", ctx); err != nil {
		// We can ignore if we fail to notify service owner
		logrus.Warnf("Failed to notify service owner for service %s", service)
	}

	autoValidateIfRequired(ctx, service)

	return resolver, nil
}

func getPrimaryOwnerEmail(userId uint) string {
	primaryOwner, err := getLDAPUserFromID(uint32(userId))
	if err != nil {
		logrus.Warnf("Failed to get old primary owner data for user with ID %d due to: %s", userId, err.Error())
		return ""
	}
	return primaryOwner.Mail
}

func (r *Resolver) UpdateService(ctx context.Context,
	args struct {
		ID      graphql.ID
		Service *serviceInput
	},
) (*serviceResolver, error) {
	// Get the existing service
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		logrus.Errorf("Failed to convert ID %d to uint due to: %s", args.ID, err.Error())
		return nil, err
	}
	service, err := catalog.GetCatalog().GetServiceByID(id)
	if err != nil {
		logrus.Errorf("Failed to query service with ID %d due to: %s", args.ID, err.Error())
		return nil, err
	}

	oldPrimaryOwnerEmail := getPrimaryOwnerEmail(service.PrimaryOwnerID)

	beforeLog := service.LogInfo()

	// Update the given fields
	if err := mergeServiceData(service, args.Service, ctx, r); err != nil {
		logrus.Errorf("Failed to merge data while updating service %s due to: %s", args.Service, err.Error())
		return nil, err
	}

	newPrimaryOwnerEmail := getPrimaryOwnerEmail(service.PrimaryOwnerID)

	// update upstream & downstream dependencies of the service
	// this updates service attributes but does not save the service
	if err := postSaveUpdates(service, args.Service, ctx, r); err != nil {
		logrus.Errorf("Failed post-save procedure while updating service %s due to: %s", args.Service, err.Error())
		return nil, err
	}

	// save service with updated attributes
	resolver, err := saveService(service)
	if err != nil {
		logrus.Errorf("Failed to persist updated attributes on while updating service %s due to: %s", args.Service, err.Error())
		return nil, err
	}

	afterLog := service.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, beforeLog, afterLog, ctx)

	var oldServiceLog catalog.ServiceLog
	err = json.Unmarshal([]byte(beforeLog.Data), &oldServiceLog)
	if err != nil {
		logrus.Errorf("Failed to decode json for oldServiceLog: %s", err)
	}

	var newServiceLog catalog.ServiceLog
	err = json.Unmarshal([]byte(afterLog.Data), &newServiceLog)
	if err != nil {
		logrus.Errorf("Failed to decode json for newServiceLog: %s", err)
	}

	// check if Primary Ownership is changing
	if oldServiceLog.PrimaryOwnerUID != newServiceLog.PrimaryOwnerUID {
		// send slack notification(s)
		if err := notifyServiceOwner(service, newServiceLog.PrimaryOwnerUID, newPrimaryOwnerEmail, newServiceLog.PrimaryOwnerName, oldServiceLog.PrimaryOwnerUID, oldPrimaryOwnerEmail, oldServiceLog.PrimaryOwnerName, ctx); err != nil {
			// We can ignore if we fail to notify service owner
			logrus.Warnf("Failed to notify service owner for service %s", service)
		}
		// auto-validate if required
		autoValidateIfRequired(ctx, service)
	} else {
		// get current authenticated user
		currentUser := goracleUser.GetUserFromContext(ctx)
		if currentUser == nil {
			// if auto-validation fails, log error and continue (do not fail flow)
			logrus.Errorf("Auto-Validation Error: Failed to retrieve current user from guardian")
			return resolver, nil
		}
		// add service state audit. always automatically validate the value (set by user) for creation and update operations
		serviceAudit := &catalog.ServiceAudit{Auditor: currentUser.UID, ServiceID: service.ID, Action: "validated", AuditType: "state", AuditValue: service.State.Name}
		serviceAudit.AuditTime = time.Now()
		err = catalog.GetCatalog().AddServiceAudit(serviceAudit)
		if err != nil {
			logrus.Errorf("Auto-Validation Error: Failed to add service audit (state) for serviceID: %d", service.ID)
		}
	}

	return resolver, nil
}

func (r *Resolver) DeleteService(ctx context.Context,
	args struct {
		ID graphql.ID
	},
) (*serviceResolver, error) {
	// Get the existing service
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		logrus.Errorf("Failed to convert ID %d to uint due to: %s", args.ID, err.Error())
		return nil, err
	}
	service, err := catalog.GetCatalog().GetServiceByID(id)
	if err != nil {
		logrus.Errorf("Failed to query service with ID %d due to: %s", args.ID, err.Error())
		return nil, err
	}

	// Delete it
	err = catalog.GetCatalog().DeleteService(service)
	if err != nil {
		logrus.Errorf("Failed to delete service with ID %d due to: %s", args.ID, err.Error())
		return nil, err
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, service, nil, ctx)

	return &serviceResolver{s: service}, nil
}

func saveService(service *catalog.Service) (*serviceResolver, error) {
	// checks:
	// - name: DB handles it
	// - type: must not be empty
	// - primary owner: must not be empty or 0
	// - "tier" attribute: mergeServiceData() handles it
	// - state: must not be empty
	if service.State == nil {
		return nil, fmt.Errorf("missing required field: 'state'")
	}
	if service.ServiceType == nil {
		return nil, fmt.Errorf("missing required field: 'serviceType'")
	}
	if service.PrimaryOwnerID == 0 {
		return nil, fmt.Errorf("must have Primary Owner assigned")
	}
	err := catalog.GetCatalog().AddService(service)
	if err != nil {
		return nil, err
	}
	return &serviceResolver{s: service}, nil
}

func notifyServiceOwner(serviceCurrent *catalog.Service, newPrimaryOwnerUID string, newPrimaryOwnerEmail string, newPrimaryOwnerCN string, oldPrimaryOwnerUID string, oldPrimaryOwnerEmail string, oldPrimaryOwnerCN string, ctx context.Context) error {
	if serviceCurrent == nil {
		return fmt.Errorf("insufficient information to send slack notification")
	}

	// create a slice for the errors
	var errStrings []string

	currentUser := goracleUser.GetUserFromContext(ctx)
	if currentUser == nil {
		return fmt.Errorf("error retrieving current user from guardian")
	}

	if oldPrimaryOwnerUID == "" {
		// trigger slack notifications for new service
		logrus.Debug("sending slack notifications for new service")
		subject := "Primary Ownership Assignment"

		// send a slack notification to new owner only when they are not the one making this change
		if currentUser.UID != newPrimaryOwnerUID {
			body := fmt.Sprintf("Hello %s, you have been assigned as Primary Owner for the new service '%s' by %s. \n\n Please take one of these actions: \n\n1) Validate to confirm service ownership or \n\n 2) Invalidate and contact %s to resolve and if needed reassign", newPrimaryOwnerCN, serviceCurrent.Name, currentUser.CN, currentUser.CN)
			err := messaging.SlackPostDM(newPrimaryOwnerUID, newPrimaryOwnerEmail, newPrimaryOwnerCN, subject, body, serviceCurrent.Name, fmt.Sprint(serviceCurrent.ID), true, nil)
			if err != nil || os.Getenv("DEBUG_SEND_EMAIL") == "true" {
				if err != nil {
					logrus.Errorf("Failed to send slack message to new primary owner for serviceID: %d, error: %s", serviceCurrent.ID, err)
					errStrings = append(errStrings, err.Error())
				}
				messaging.SendEmailNotification(newPrimaryOwnerUID, nil, currentUser.CN, subject, body)
			}
		}
	} else {
		// trigger slack notifications for existing service
		logrus.Debug("sending slack notifications for existing service")
		subject := "Primary Ownership Transfer"

		var err1 error
		var err2 error
		// send a slack notification to new owner only when they are not the one making this change
		if currentUser.UID != newPrimaryOwnerUID {
			body := fmt.Sprintf("Hello %s, you have been assigned as Primary Owner for '%s' by %s. \n\n Please take one of these actions: \n\n1) Validate to confirm service ownership or \n\n 2) Invalidate and contact %s to resolve and if needed reassign", newPrimaryOwnerCN, serviceCurrent.Name, currentUser.CN, currentUser.CN)
			err1 = messaging.SlackPostDM(newPrimaryOwnerUID, newPrimaryOwnerEmail, newPrimaryOwnerCN, subject, body, serviceCurrent.Name, fmt.Sprint(serviceCurrent.ID), true, nil)
			if err1 != nil {
				logrus.Errorf("Failed to send slack message to new primary owner for serviceID: %d, error: %s", serviceCurrent.ID, err1)
				errStrings = append(errStrings, err1.Error())
			}
		}

		// send a slack notification to previous owner only when they are not the one making this change
		if currentUser.UID != oldPrimaryOwnerUID {
			body := fmt.Sprintf("Hello %s, you have been un-assigned as Primary Owner for '%s' by %s. \n\n If you have any concerns about this change, please reach out to %s", oldPrimaryOwnerCN, serviceCurrent.Name, currentUser.CN, currentUser.CN)
			err2 = messaging.SlackPostDM(oldPrimaryOwnerUID, oldPrimaryOwnerEmail, oldPrimaryOwnerCN, subject, body, serviceCurrent.Name, fmt.Sprint(serviceCurrent.ID), false, nil)
			if err2 != nil {
				logrus.Errorf("Failed to send slack message to old primary owner for serviceID: %d, error: %s", serviceCurrent.ID, err2)
				errStrings = append(errStrings, err2.Error())
			}
		}

		if (err1 != nil || err2 != nil) || os.Getenv("DEBUG_SEND_EMAIL") == "true" {
			// the 2 slack messages above may fail *independently*
			// the email is always to the new primary owner, and cc'ed to all associated parties
			body := fmt.Sprintf("Hello %s, you have been assigned as Primary Owner for '%s' by %s. \n\n Please take one of these actions: \n\n1) Validate to confirm service ownership or \n\n 2) Invalidate and contact %s to resolve and if needed reassign", newPrimaryOwnerCN, serviceCurrent.Name, currentUser.CN, currentUser.CN)
			messaging.SendEmailNotification(newPrimaryOwnerUID, []string{oldPrimaryOwnerUID, currentUser.UID}, currentUser.CN, subject, body)
		}

	}

	if len(errStrings) > 0 {
		return errors.New( "slack error(s): " + strings.Join(errStrings, ", "))
	}

	return nil
}

func mergeServiceData(dbSvc *catalog.Service, apiSvc *serviceInput, ctx context.Context, r *Resolver) error {
	if apiSvc.Name != nil {
		dbSvc.Name = *apiSvc.Name
	}
	if apiSvc.Description != nil {
		dbSvc.Description = *apiSvc.Description
	}
	if apiSvc.AvailabilityObjective != nil {
		dbSvc.AvailabilityObjective = *apiSvc.AvailabilityObjective
	}
	if apiSvc.Environment != nil {
		dbSvc.Environment = *apiSvc.Environment
	}
	if apiSvc.Known != nil {
		dbSvc.Known = *apiSvc.Known
	}
	if apiSvc.Type != nil {
		serviceTypes, err := catalog.GetServiceTypeMap()
		if err != nil {
			return err
		}
		serviceType, found := serviceTypes[*apiSvc.Type]
		if !found {
			return fmt.Errorf("no such service type '%s'", *apiSvc.Type)
		}
		// This should be temporary until we can get rid of the
		// ServiceType table
		dbSvc.ServiceTypeID = serviceType.ID
		dbSvc.ServiceType = serviceType
	}
	if apiSvc.State != nil {
		states, err := catalog.GetServiceStateMap()
		if err != nil {
			return err
		}
		state, found := states[strings.ToLower(*apiSvc.State)]
		if !found {
			return fmt.Errorf("no such service state '%s'", *apiSvc.State)
		}
		dbSvc.StateID = state.ID
		dbSvc.State = state
	}
	if apiSvc.Components != nil {
		comps := []*catalog.Component{}
		for _, comp := range *apiSvc.Components {
			if comp.ID == nil {
				args := CreateComponentArgs{Component: &comp}
				res, err := r.CreateComponent(ctx, args)
				if err != nil {
					return err
				}
				comps = append(comps, res.c)
			} else {
				id, err := idStringToUint(*comp.ID)
				if err != nil {
					return err
				}
				if id == 0 {
					args := CreateComponentArgs{Component: &comp}
					res, err := r.CreateComponent(ctx, args)
					if err != nil {
						return err
					}
					comps = append(comps, res.c)
				} else {
					args := UpdateComponentArgs{ID: *comp.ID, Component: &comp}
					res, err := r.UpdateComponent(ctx, args)
					if err != nil {
						return err
					}
					comps = append(comps, res.c)
				}
			}
		}
		dbSvc.Components = comps
	} else if apiSvc.ComponentIDs != nil {
		dbSvc.Components = make([]*catalog.Component, 0)
		for _, cid := range *apiSvc.ComponentIDs {
			comp, err := resolveComponent(cid)
			if err != nil {
				return err
			}
			if comp != nil {
				dbSvc.Components = append(dbSvc.Components, comp.c)
			}
		}
	}

	if apiSvc.PagerDuty != nil {
		dbSvc.PagerDuty = *apiSvc.PagerDuty
	}

	// backwards compat, remove later
	// TODO(mmicch) Remove after slack transition
	if apiSvc.Slack != nil {
		dbSvc.Slack = *apiSvc.Slack
	}

	if apiSvc.SlackChannelID != nil {
		dbSvc.Slack = string(*apiSvc.SlackChannelID)
	}

	if apiSvc.PrimaryOwnerID != nil || apiSvc.PrimaryOwnerUID != nil {
        var employeeNumber uint
        var err error
        // DO NOT allow setting the primary owner ID back to 0 (unassociated)
        primaryOwnerIDZero := apiSvc.PrimaryOwnerID != nil && (*apiSvc.PrimaryOwnerID == "" || *apiSvc.PrimaryOwnerID == graphql.ID("0"))
        primaryOwnerUIDZero := apiSvc.PrimaryOwnerUID != nil && (*apiSvc.PrimaryOwnerUID == "" || *apiSvc.PrimaryOwnerUID == "0")
        if primaryOwnerIDZero || primaryOwnerUIDZero {
            return fmt.Errorf("Cannot set Primary Owner to empty!")
        } else {
            if apiSvc.PrimaryOwnerUID != nil {
                c, err := ldapLibrary.NewClient()
                if err != nil {
                    return err
                }
                user, err := c.GetUserInfoByName(*apiSvc.PrimaryOwnerUID)
                if err != nil {
                    return err
                }
                if user == nil {
                    return fmt.Errorf("could not find UID %v in LDAP", user)
                }
                employeeNumber = uint(user.EmployeeNumber)
            } else {
                // Don't update this field unless the user is legit
                employeeNumber, err = validatePrimaryOwnerID(*apiSvc.PrimaryOwnerID)
                if err != nil {
                    return fmt.Errorf("Could not resolve given primary_owner_id: " + err.Error())
                }
            }
        }

        dbSvc.PrimaryOwnerID = employeeNumber
	}

	// in both create and update: input must contain 'tier' attribute
	foundRequiredAttr := false
	if apiSvc.Attributes != nil {
		for _, attr := range *apiSvc.Attributes {
			if attr.Name == "tier" {
				foundRequiredAttr = true
				break
			}
		}
	}

	if !foundRequiredAttr && dbSvc.ID != 0 { // if update, find existing tier attribute
		params := make(map[string]interface{})
		params["object_type"] = catalog.LogTypeService
		params["object_id"] = dbSvc.ID
		attrs, err := catalog.GetCatalog().GetAttributes(params)
		if err != nil {
			return err
		}
		for _, attr := range attrs {
			if attr.Name == "tier" {
				foundRequiredAttr = true
				break
			}
		}
	}

	if !foundRequiredAttr {
		return fmt.Errorf("missing required field: attribute 'tier'")
	}

	return nil
}

func postSaveUpdates(dbSvc *catalog.Service, apiSvc *serviceInput, ctx context.Context, r *Resolver) error {
	// Update service upstreams if set
	if apiSvc.ServiceUpstreams != nil {
		// Get a list of existing service upstreams
		var sdArgs struct {
			RootServiceID       *graphql.ID
			DownstreamServiceID *graphql.ID
			AttributeName       *string
			AttributeValue      *string
		}

		id := idUintToString(dbSvc.ID)
		sdArgs.DownstreamServiceID = &id
		oldUpstreams, _ := r.ServiceDependencies(sdArgs)

		// Delete old upstreams not on the list
		for _, oldSD := range oldUpstreams {
			found := false
			for _, newSD := range *apiSvc.ServiceUpstreams {
				if newSD.ID != nil && *newSD.ID == oldSD.ID() {
					found = true
					break
				}
			}
			if !found {
				var args struct {
					ID graphql.ID
				}
				args.ID = oldSD.ID()
				r.DeleteServiceDependency(ctx, args)
			} else {
				// FIXME: Implement update at some point
			}
		}

		// Add new upstreams
		for _, sd := range *apiSvc.ServiceUpstreams {
			if sd.ID == nil || *sd.ID == "0" {
				sd.DownstreamServiceID = &id
				if sd.DownstreamService != nil {
					sd.DownstreamService.ID = &id
				}
				var args struct{ ServiceDependency *serviceDependencyInput }
				args.ServiceDependency = sd
				_, err := r.CreateServiceDependency(ctx, args)
				if err != nil {
					return err
				}
			}
		}

	}

	// Update service downstreams if set
	if apiSvc.ServiceDownstreams != nil {
		// Get a list of existing service upstreams
		var sdArgs struct {
			RootServiceID       *graphql.ID
			DownstreamServiceID *graphql.ID
			AttributeName       *string
			AttributeValue      *string
		}

		id := idUintToString(dbSvc.ID)
		sdArgs.RootServiceID = &id
		oldDownstreams, _ := r.ServiceDependencies(sdArgs)

		// Delete old downstreams not on the list
		for _, oldSD := range oldDownstreams {
			found := false
			for _, newSD := range *apiSvc.ServiceDownstreams {
				if newSD.ID != nil && *newSD.ID == oldSD.ID() {
					found = true
					break
				}
			}
			if !found {
				var args struct {
					ID graphql.ID
				}
				args.ID = oldSD.ID()
				r.DeleteServiceDependency(ctx, args)
			} else {
				// FIXME: Implement update at some point
			}
		}

		// Add new downstreams
		for _, sd := range *apiSvc.ServiceDownstreams {
			if sd.ID == nil || *sd.ID == "0" {
				sd.RootServiceID = &id
				if sd.RootService != nil {
					sd.RootService.ID = &id
				}
				var args struct{ ServiceDependency *serviceDependencyInput }
				args.ServiceDependency = sd
				_, err := r.CreateServiceDependency(ctx, args)
				if err != nil {
					return err
				}
			}
		}

	}
	r.saveAttributes(apiSvc.Attributes, idUintToString(dbSvc.ID), catalog.LogTypeService, ctx)
	return nil
}

func (r *serviceResolver) Logs(
	args struct {
		Offset *int32
		Limit  *int32
	}) []*logRecordResolver {

	offset := -1
	limit := -1

	if args.Offset != nil {
		offset = int(*args.Offset)
	}
	if args.Limit != nil {
		limit = int(*args.Limit)
	}

	res, err := resolveLogs("service", r.s.ID, offset, limit)
	if err != nil {
		logrus.Errorf("Failed to resolve logs for service with ID %d due to: %s", r.s.ID, err.Error())
		return nil
	}
	return res
}
