package apiv2

import (
	"context"

	graphql "github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/catalog"
	"github.com/sirupsen/logrus"
)

type serviceDependencyResolver struct {
	sd *catalog.ServiceDependency
}

func (r *Resolver) ServiceDependencies(args struct {
	RootServiceID       *graphql.ID
	DownstreamServiceID *graphql.ID
	AttributeName       *string
	AttributeValue      *string
}) ([]*serviceDependencyResolver, error) {
	ids, err := idsWithAttribute(catalog.LogTypeServiceDependency, args.AttributeName, args.AttributeValue)
	if err != nil {
		return nil, err
	}
	params := make(map[string]interface{})
	if args.RootServiceID != nil {
		rootID, err := idStringToUint(*args.RootServiceID)
		if err != nil {
			return nil, err
		}
		params["root_service_id"] = rootID
	}
	if args.DownstreamServiceID != nil {
		downID, err := idStringToUint(*args.DownstreamServiceID)
		if err != nil {
			return nil, err
		}
		params["downstream_service_id"] = downID
	}
	servDeps, err := catalog.GetCatalog().GetServiceDependenciesComplete(ids, params)
	if err != nil {
		return nil, err
	}

	serviceDependencyResolvers := []*serviceDependencyResolver{}
	for _, sd := range servDeps {
		serviceDependencyResolvers = append(serviceDependencyResolvers, &serviceDependencyResolver{sd: sd})
	}
	return serviceDependencyResolvers, nil
}

func resolveServiceDependencies(rootServiceID *uint, downstreamServiceID *uint) ([]*serviceDependencyResolver, error) {
	params := make(map[string]interface{})
	if rootServiceID != nil {
		params["root_service_id"] = *rootServiceID
	}
	if downstreamServiceID != nil {
		params["downstream_service_id"] = *downstreamServiceID
	}

	servDeps, err := catalog.GetCatalog().GetServiceDependencies(params)
	if err != nil {
		return nil, err
	}
	serviceDependencyResolvers := []*serviceDependencyResolver{}
	for _, servDep := range servDeps {
		resolver := serviceDependencyResolver{sd: servDep}
		if resolver.RootService() == nil || resolver.DownstreamService() == nil {
			logrus.Warn("Deleting Dep. attached service not found for dep %v", servDep)
			catalog.GetCatalog().DeleteServiceDependency(servDep)
			continue
		}
		serviceDependencyResolvers = append(serviceDependencyResolvers, &resolver)
	}
	return serviceDependencyResolvers, nil
}

func (r *serviceDependencyResolver) ID() graphql.ID {
	return idUintToString(r.sd.ID)
}

func (r *serviceDependencyResolver) RootServiceID() graphql.ID {
	return idUintToString(r.sd.RootServiceID)
}

func (r *serviceDependencyResolver) RootService() *serviceResolver {
	res, err := resolveService(r.RootServiceID())
	if err != nil {
		return nil
	}
	return res
}

func (r *serviceDependencyResolver) DownstreamServiceID() graphql.ID {
	return idUintToString(r.sd.DownstreamServiceID)
}

func (r *serviceDependencyResolver) DownstreamService() *serviceResolver {
	res, err := resolveService(r.DownstreamServiceID())
	if err != nil {
		return nil
	}
	return res
}

func (r *serviceDependencyResolver) Attributes() []*attributeResolver {
	res, err := resolveAttributes(catalog.LogTypeServiceDependency, r.sd.ID)
	if err != nil {
		return nil
	}
	return res
}

// Mutations
type serviceDependencyInput struct {
	ID                  *graphql.ID        `json:"id"`
	RootServiceID       *graphql.ID        `json:"root_service_id"`
	DownstreamServiceID *graphql.ID        `json:"downstream_service_id"`
	Attributes          *[]*attributeInput `json:"-"`

	// Used for parsing incoming, not actually ever used for mutating directly
	RootService       *serviceInput `json:"root_service"`
	DownstreamService *serviceInput `json:"downstream_service"`
}

func (r *Resolver) CreateServiceDependency(ctx context.Context,
	args struct {
		ServiceDependency *serviceDependencyInput
	},
) (*serviceDependencyResolver, error) {

	// Create struct
	servDep := &catalog.ServiceDependency{}

	if err := mergeServiceDependencyData(servDep, args.ServiceDependency); err != nil {
		return nil, err
	}
	resolver, err := saveServiceDependency(servDep)
	if err != nil {
		return nil, err
	}
	if args.ServiceDependency.Attributes != nil {
		for _, attr := range *args.ServiceDependency.Attributes {
			attr.ObjectID = resolver.ID()
			attr.ObjectType = catalog.LogTypeServiceDependency
			var args struct{ Attribute *attributeInput }
			args.Attribute = attr
			r.CreateAttribute(ctx, args)
		}
	}
	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, resolver.sd, ctx)

	return resolver, nil
}

func (r *Resolver) DeleteServiceDependency(ctx context.Context,
	args struct {
		ID graphql.ID
	},
) (*serviceDependencyResolver, error) {

	// Get the existing serviceDependency
	var id uint
	var err error
	if id, err = idStringToUint(args.ID); err != nil {
		return nil, err
	}
	serviceDependency, err := catalog.GetCatalog().GetServiceDependencyByID(id)

	// Delete it
	err = catalog.GetCatalog().DeleteServiceDependency(serviceDependency)
	if err != nil {
		return nil, err
	}
	catalog.SaveAPILoggables(catalog.LogOpDelete, serviceDependency, nil, ctx)

	return &serviceDependencyResolver{sd: serviceDependency}, nil
}

func mergeServiceDependencyData(dbServDep *catalog.ServiceDependency, apiServDep *serviceDependencyInput) error {
	if apiServDep.RootService != nil {
		rootRes, err := resolveService(*apiServDep.RootService.ID)
		if err != nil {
			return err
		}
		dbServDep.RootServiceID = rootRes.s.ID
	} else {
		rootRes, err := resolveService(*apiServDep.RootServiceID)
		if err != nil {
			return err
		}
		dbServDep.RootServiceID = rootRes.s.ID
	}

	if apiServDep.DownstreamService != nil {
		downRes, err := resolveService(*apiServDep.DownstreamService.ID)
		if err != nil {
			return err
		}
		dbServDep.DownstreamServiceID = downRes.s.ID
	} else {
		downRes, err := resolveService(*apiServDep.DownstreamServiceID)
		if err != nil {
			return err
		}
		dbServDep.DownstreamServiceID = downRes.s.ID

	}
	return nil
}

func saveServiceDependency(servDep *catalog.ServiceDependency) (*serviceDependencyResolver, error) {
	err := catalog.GetCatalog().AddServiceDependency(servDep)
	if err != nil {
		return nil, err
	}
	return &serviceDependencyResolver{sd: servDep}, nil
}
