package apiv2

import (
	"strings"

	"code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/messaging"
	graphql "github.com/neelance/graphql-go"
	"github.com/nlopes/slack"
	"github.com/sirupsen/logrus"
)

type slackResolver struct {
	client *slack.Client
}

type slackUserResolver struct {
	slackUser slack.User
}

func (r *slackUserResolver) ID() graphql.ID {
	return graphql.ID(r.slackUser.ID)
}

func (r *slackUserResolver) Name() string {
	return r.slackUser.Name
}

func (r *slackUserResolver) RealName() string {
	return r.slackUser.RealName
}

type slackChannelResolver struct {
	slackChannel slack.Channel
}

func (r *slackChannelResolver) ID() graphql.ID {
	return graphql.ID(r.slackChannel.ID)
}

func (r *slackChannelResolver) Name() string {
	return r.slackChannel.Name
}

func (r *slackChannelResolver) Topic() string {
	return r.slackChannel.Topic.Value
}

func (r *slackChannelResolver) Purpose() string {
	return r.slackChannel.Purpose.Value
}

func (r *Resolver) Slack() (*slackResolver, error) {
	client := slack.New(config.Config.SlackToken)
	_, err := client.AuthTest()
	if err != nil {
		logrus.Errorf("failed slack client auth test due to: %s", err.Error())
		return nil, err
	}

	return &slackResolver{
		client: client,
	}, nil
}

func (r *slackResolver) Users() ([]*slackUserResolver, error) {
	users, err := r.client.GetUsers()
	if err != nil {
		logrus.Errorf("failed to get list of users due to: %s", err.Error())
		return nil, err
	}
	userResolvers := make([]*slackUserResolver, 0)
	for _, u := range users {
		if u.Deleted || u.IsBot {
			continue
		}
		userResolvers = append(userResolvers, &slackUserResolver{slackUser: u})
	}
	return userResolvers, nil
}

func (r *slackResolver) Channels() ([]*slackChannelResolver, error) {
	channels, err := r.client.GetChannels(true)
	if err != nil {
		logrus.Errorf("failed to get list of channels due to: %s", err.Error())
		return nil, err
	}
	chanResolvers := make([]*slackChannelResolver, 0)
	for _, ch := range channels {
		chanResolvers = append(chanResolvers, &slackChannelResolver{slackChannel: ch})
	}
	return chanResolvers, nil
}

func resolveSlackChannel(id graphql.ID) (*slackChannelResolver, error) {
	client := slack.New(config.Config.SlackToken)
	_, err := client.AuthTest()
	if err != nil {
		return nil, err
	}
	// backwards compat, remove later
	channelID := string(id)
	if messaging.IsSlackChannelName(channelID) {
		chName := strings.TrimLeft(channelID, "#")
		channelID, err = messaging.GetSlackChannelIDFromName(chName, client)
		if err != nil {
			return nil, err
		}
	}
	// end of backwards compat
	channel, err := messaging.GetSlackChannelInfo(channelID, client)
	if err != nil {
		return nil, err
	}
	return &slackChannelResolver{slackChannel: channel}, nil
}
