package apiv2

import (
	graphql "github.com/neelance/graphql-go"

	"code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/messaging"
	"errors"
	"github.com/nlopes/slack"
	"github.com/sirupsen/logrus"
	"sync"
	"time"
    "code.justin.tv/qe/twitchldap"
)

const (
	ACTIVE  = "active"
	MISSING = "missing"
	ERROR   = "error"
	ldapCacheExpirationHour = 12
)

type userResolver struct {
    info *twitchldap.EmployeeInfo
	err  error
}

func (r *Resolver) User(args struct{ EmployeeNumber graphql.ID }) (*userResolver, error) {
	return resolveUser(args.EmployeeNumber)
}

// resolves a user with the LDAP information. Does this on a best
// effort basis, and will embed the error within the user info
// if any error occurs.  this will swallow all errors to prevent
// an LDAP outage from causing API outages
// the method signature still reports retuning an error just to
// match convention and maybe someday use implement a way to
// properly return errors to the caller
func resolveUser(id graphql.ID) (*userResolver, error) {
	employeeNumber, err := idStringToUint(id)
	if err != nil {
		logrus.Warnf("Given ldap id to resolve that is not an int %v", id)
	}
	// info can be nil if the user is not found
	info, err := getLDAPUserFromID(uint32(employeeNumber))
	if err != nil {
		logrus.Errorf("failed to get ldap user with employee ID %s due to: %s", id, err.Error())
		return &userResolver{info: nil, err: err}, nil
	}
	return &userResolver{info: info}, nil
}

func (r *Resolver) Users() ([]*userResolver, error) {
	return resolveAllUsers()
}

func resolveAllUsers() ([]*userResolver, error) {
	usersInfo, err := getAllLDAPUsers()
	if err != nil {
		logrus.Errorf("failed to get list of users from ldap due to: %s", err.Error())
		return nil, err
	}
	userResolvers := make([]*userResolver, 0)
	for _, info := range usersInfo {
		userResolvers = append(userResolvers, &userResolver{info: info})
	}
	return userResolvers, nil
}

var ldapUsersMutex sync.RWMutex
var ldapUsersCacheLastUpdated time.Time
var ldapUserIDToUserInfo map[uint32]*twitchldap.EmployeeInfo
var ldapUsers []*twitchldap.EmployeeInfo

func getAllLDAPUsers() ([]*twitchldap.EmployeeInfo, error) {
	err := refreshLdapCacheIfOld()
	if err != nil {
		return nil, err
	}
	ldapUsersMutex.RLock()
	defer ldapUsersMutex.RUnlock()
	if ldapUsers == nil {
		logrus.Warn("Encountered a nil ldap cache after refresh, something is probably wrong")
	}
	return ldapUsers, nil
}

func getLDAPUserFromID(employeeID uint32) (*twitchldap.EmployeeInfo, error) {
	err := refreshLdapCacheIfOld()
	if err != nil {
		return nil, err
	}
	ldapUsersMutex.RLock()
	defer ldapUsersMutex.RUnlock()
	user, ok := ldapUserIDToUserInfo[employeeID]
	if !ok {
		return nil, errors.New("could not find the given employee in LDAP")
	}
	return user, nil
}

func refreshLdapCacheIfOld() error {
	if ldapUserIDToUserInfo != nil && time.Now().Sub(ldapUsersCacheLastUpdated).Hours() < ldapCacheExpirationHour {
		return nil
	}

	ldapUsersMutex.Lock()
	defer ldapUsersMutex.Unlock()
	if ldapUserIDToUserInfo != nil && time.Now().Sub(ldapUsersCacheLastUpdated).Hours() < ldapCacheExpirationHour {
		return nil
	}
	c, err := twitchldap.NewClient()
	defer c.Close()
	if err != nil {
		return err
	}
	users, err := c.GetAllUserInfo()
	if err != nil {
		return err
	}

	logrus.Debug("Refreshing ldap user caches")
	ldapUserIDToUserInfo = make(map[uint32]*twitchldap.EmployeeInfo)
	ldapUsers = nil
	for _, user := range users {
		ldapUsers = append(ldapUsers, user)
		ldapUserIDToUserInfo[user.EmployeeNumber] = user
	}
	ldapUsersCacheLastUpdated = time.Now()
	return nil
}

func (r *userResolver) CN() *string {
	if r.info != nil {
		return &r.info.CN
	}
	return nil
}

func (r *userResolver) UID() *string {
	if r.info != nil {
		return &r.info.UID
	}
	return nil
}

func (r *userResolver) Mail() *string {
	if r.info != nil {
		return &r.info.Mail
	}
	return nil
}

func (r *userResolver) Surname() *string {
	if r.info != nil {
		return &r.info.Surname
	}
	return nil
}

func (r *userResolver) GivenName() *string {
	if r.info != nil {
		return &r.info.GivenName
	}
	return nil
}

func (r *userResolver) PreferredName() *string {
	if r.info != nil {
		return &r.info.PreferredName
	}
	return nil
}

func (r *userResolver) Inactive() *bool {
    if r.info != nil {
        return &r.info.Inactive
    }
    return nil
}

func (r *userResolver) EmployeeNumber() *graphql.ID {
	if r.info != nil {
		id := idUintToString(uint(r.info.EmployeeNumber))
		return &id
	}
	return nil
}

func (r *userResolver) AmazonUID() *string {
	if r.info != nil {
        return &r.info.TwitchAmznUID
	}
	return nil
}

func (r *userResolver) Department() *string {
	if r.info != nil {
		return &r.info.Department
	}
	return nil
}

func (r *userResolver) Manager() *string {
	if r.info != nil {
        return &r.info.ManagerLDAP
	}
	return nil
}

func (r *userResolver) BuildingName() *string {
	if r.info != nil {
		return &r.info.BuildingName
	}
	return nil
}

func (r *userResolver) Slack() *slackUserResolver {
	client := slack.New(config.Config.SlackToken)
	// Currently slack has a bug where bot tokens cannot use getuserbyemail.
	//var user *slack.User
	//// attempt to retrieve user info by inconsistent email address usage
	//user, err := client.GetUserByEmail(targetUser + "@justin.tv")
	//if err == nil {
	//	return &slackUserResolver{*user}
	//}
	//user, err = client.GetUserByEmail(targetUser + "@twitch.tv")
	//if err == nil {
	//	return &slackUserResolver{*user}
	//}
	// worst case -- fallback is to search through user list -- GetUserByEmail() can fail due to various errors
	var user slack.User
	user, err := messaging.GetSlackUser(*r.UID(), *r.Mail(), *r.CN(), client)
	if err != nil {
		logrus.Errorf("failed to get slack user with user ID %s, username %s due to: %s", *r.UID(), *r.CN(), err.Error())
		return nil
	}
	return &slackUserResolver{user}
}

func (r *userResolver) Status() string {
	if r.info == nil {
		return MISSING
	}
	if r.err != nil {
		return ERROR
	}
	// TODO: add a check to see if the user is "inactive"
	// but this isn't currently possible
	return ACTIVE
}

func (r *userResolver) Error() *string {
	if r.info != nil && r.err != nil {
		errMsg := r.err.Error()
		return &errMsg
	}
	return nil
}

// Mutations
// This type has no mutations because there should be no need to create
// users explicitly.  This catalog table is simply a mirror from LDAP,
// and the only reason a table is needed is for services to reference
// user IDs as primary service owners.
//
// Users will only be added to the catalog when they are assigned as owner
// of a service. The service API calls will handle the logic there.
