package catalog

import (
	"testing"
	"time"

	"github.com/jinzhu/gorm"
	_ "github.com/mattn/go-sqlite3"
	"github.com/stretchr/testify/assert"
)

var c *dbCatalog
var stype *ServiceType

func init() {
	dsn := "sqlite3://file::memory:?mode=memory&cache=shared"
	SetupDB(dsn)
	c = getDBCatalog()
	AddBaseServiceTypes()

	stmap, err := GetServiceTypeMap()
	if err != nil {
		panic(err)
	}
	st, ok := stmap["internal-ops"]
	if !ok {
		panic("Couldn't lookup internal ops service")
	}
	stype = st
}

func TestDbCatalogService(t *testing.T) {
	s1 := DefaultService()
	s1.Name = "foo"
	s1.ServiceType = stype

	err := c.AddService(s1)
	assert.NoError(t, err, "AddService failed")

	sgot1, err := c.GetServiceByID(s1.ID)
	assert.NoError(t, err, "GetServiceByID failed")

	assert.Equal(t, sgot1.ID, s1.ID, "GetServiceByID ID doesn't match")
	assert.Equal(t, sgot1.Name, s1.Name, "GetServiceByID Name doesn't match")

	// dupe test
	s2 := &Service{
		Name:        "foo",
		ServiceType: stype,
	}
	err = c.AddService(s2)
	assert.Error(t, err, "duplicate AddService should fail")

	// delete
	s3 := &Service{}
	s3.ID = s1.ID
	// s3.Name = "foo"

	err = c.DeleteService(s3)
	assert.NoError(t, err, "DeleteService failed")

	_, err = c.GetServiceByID(s1.ID)

	assert.Error(t, err, "GetServiceByID shouldn't succeed after delete")

	// re-add after delete
	s4 := &Service{
		Name:        "foo",
		ServiceType: stype,
	}
	err = c.AddService(s4)
	assert.NoError(t, err, "service AddService after delete should succeed")
}

func TestDbCatalogComponent(t *testing.T) {
	c1 := &Component{
		Name:  "foo",
		Label: "bar",
		Type:  "beanstalk",
	}

	err := c.AddComponent(c1)
	assert.NoError(t, err, "AddComponent failed")

	c2 := &Component{
		Name:  "foo",
		Label: "bar",
		Type:  "beanstalk",
	}

	err = c.AddComponent(c2)
	assert.Error(t, err, "duplicate AddComponent should fail")

	cgot1, err := c.GetComponentByID(c1.ID)
	assert.NoError(t, err, "GetComponentByID failed")
	assert.Equal(t, cgot1.ID, c1.ID, "GetComponentByID ID doesn't match")
	assert.Equal(t, cgot1.Label, c1.Label, "GetComponentByID Name doesn't match")

	c3 := &Component{
		Model: gorm.Model{ID: c1.ID},
	}
	err = c.DeleteComponent(c3)
	assert.NoError(t, err, "DeleteComponent failed")

	_, err = c.GetComponentByID(c1.ID)
	assert.Error(t, err, "GetComponentByID shouldn't succeed after Delete")

	c4 := &Component{
		Name:  "foo",
		Label: "bar",
		Type:  "beanstalk",
	}

	err = c.AddComponent(c4)
	assert.Error(t, err, "AddComponent shouldn't succeed after Delete")
}

func TestDBCatalogAttribute(t *testing.T) {
	a1 := &Attribute{
		Name:       "fooorg",
		ObjectType: "footype",
		ObjectId:   1,
	}

	err := c.AddAttribute(a1)
	assert.NoError(t, err, "AddAttribute failed")

	a2 := &Attribute{
		Name:       "fooorg",
		ObjectType: "footype",
		ObjectId:   1,
	}

	err = c.AddAttribute(a2)
	assert.Error(t, err, "duplicate AddAttribute should fail")

	agot, err := c.GetAttributes(nil)
	assert.NoError(t, err, "GetOrgByID failed")
	assert.Len(t, agot, 1, "There should only be one attribute")
	assert.Equal(t, agot[0].ID, a1.ID, "GetAttributes ID doesn't match")
	assert.Equal(t, agot[0].Name, a1.Name, "GetAttributes Name doesn't match")
	assert.Equal(t, agot[0].ObjectType, a1.ObjectType, "GetAttributes ObjectType doesn't match")
	assert.Equal(t, agot[0].ObjectId, a1.ObjectId, "GetAttributes ObjectId doesn't match")

	err = c.DeleteAttribute(a1)
	assert.NoError(t, err, "DeleteAttribute should succeed")
}

func TestDbCatalogComplex(t *testing.T) {
	//addquery
	q1 := &Query{
		Query:         "query1",
		AggregateType: AggregateType("average"),
	}
	err := c.AddQuery(q1)
	assert.NoError(t, err, "AddQuery failed")

	//addmetric
	m1 := &Metric{
		Label:   "metric1",
		Queries: []*Query{q1},
	}
	err = c.AddMetric(m1)
	assert.NoError(t, err, "AddMetric failed")

	//addcomponent
	c1 := &Component{
		Name:    "compname1",
		Label:   "comp1",
		Type:    "type1",
		Metrics: []*Metric{m1},
	}
	err = c.AddComponent(c1)
	assert.NoError(t, err, "AddComponent failed")

	//addservice
	s1 := &Service{
		Name:        "service1",
		ServiceType: stype,
		Components:  []*Component{c1},
	}
	err = c.AddService(s1)
	assert.NoError(t, err, "AddService failed")

	//add service audits
	saComponent := &ServiceAudit{
		AuditType: "component",
		ServiceID: s1.ID,
		AuditTime: time.Now(),
		Auditor:   "nherson",
		Action:    "validated",
	}
	saOwner := &ServiceAudit{
		AuditType: "owner",
		ServiceID: s1.ID,
		AuditTime: time.Now(),
		Auditor:   "nherson",
		Action:    "validated",
	}
	saAvailability := &ServiceAudit{
		AuditType: "availability",
		ServiceID: s1.ID,
		AuditTime: time.Now().Add(-10 * time.Minute),
		Auditor:   "nherson",
		Action:    "validated",
	}
	saAvailabilityNew := &ServiceAudit{
		AuditType: "availability",
		ServiceID: s1.ID,
		AuditTime: time.Now(),
		Auditor:   "nherson",
		Action:    "validated",
	}
	saState := &ServiceAudit{
		AuditType: "state",
		ServiceID: s1.ID,
		AuditTime: time.Now(),
		Auditor:   "lukemng",
		Action:    "validated",
	}
	err = c.AddServiceAudit(saState)
	assert.NoError(t, err, "AddServiceAudit failed")
	err = c.AddServiceAudit(saComponent)
	assert.NoError(t, err, "AddServiceAudit failed")
	err = c.AddServiceAudit(saOwner)
	assert.NoError(t, err, "AddServiceAudit failed")
	err = c.AddServiceAudit(saAvailability)
	assert.NoError(t, err, "AddServiceAudit failed")
	err = c.AddServiceAudit(saAvailabilityNew)
	assert.NoError(t, err, "AddServiceAudit failed")
	saGot := c.GetServiceAudits(map[string]interface{}{"service_id": s1.ID})
	assert.Equal(t, len(saGot), 5, "GetServiceAudits returned wrong number of audits")
	saGot = c.GetLatestServiceAudits(map[string]interface{}{"service_id": s1.ID})
	assert.Equal(t, len(saGot), 4, "GetServiceAudits returned wrong number of audits")
	//make sure that the received types make sense
	typeMap := make(map[string]int)
	for _, audit := range saGot {
		if val, found := typeMap[audit.AuditType]; found {
			typeMap[audit.AuditType] = val + 1
		} else {
			typeMap[audit.AuditType] = 1
		}
	}
	assert.Equal(t, map[string]int{"component": 1, "owner": 1, "availability": 1, "state": 1}, typeMap)

	// retreive and compare
	sgot1, err := c.GetServiceByID(s1.ID)
	assert.NoError(t, err, "GetServiceByID failed")

	qgot1 := sgot1.Components[0].Metrics[0].Queries[0].Query
	assert.Equal(t, qgot1, "query1", "Retreived query doesn't match")

	// delete Service while in use
	err = c.DeleteService(s1)
	assert.NoError(t, err, "Should be able to delete service")

	cgot1, err := c.GetComponentByID(c1.ID)
	assert.NoError(t, err, "GetComponentById Failed")
	assert.Equal(t, uint(0), cgot1.ServiceID, "service should no longer be associated to component")

	// delete component that has a metric
	err = c.DeleteComponent(c1)
	assert.NoError(t, err, "Should be able to delete Component")

	mgot1, err := c.GetMetricByID(m1.ID)
	assert.NoError(t, err, "GetMetricById Failed")
	assert.Equal(t, uint(0), mgot1.ComponentID, "component should no longer be associated to metric")

}

func TestDbCatalogServiceDendency(t *testing.T) {
	sd1 := &ServiceDependency{RootServiceID: 1, DownstreamServiceID: 2}

	err := c.AddServiceDependency(sd1)
	assert.NoError(t, err, "AddServiceDependency failed")

	sdgot1, err := c.GetServiceDependencyByID(sd1.ID)
	assert.NoError(t, err, "GetServiceByID failed")

	assert.Equal(t, sdgot1.RootServiceID, sd1.RootServiceID, "GetServiceByID ID doesn't match")
	assert.Equal(t, sdgot1.DownstreamServiceID, sd1.DownstreamServiceID, "GetServiceByID Name doesn't match")

	// dupe test
	sd2 := &ServiceDependency{RootServiceID: 1, DownstreamServiceID: 2}
	err = c.AddServiceDependency(sd2)
	assert.NoError(t, err, "duplicate AddServiceDependency should Pass")

	// delete
	sd3 := &ServiceDependency{}
	sd3.ID = sd1.ID
	// s3.Name = "foo"

	err = c.DeleteServiceDependency(sd3)
	assert.NoError(t, err, "DeleteService failed")

	_, err = c.GetServiceByID(sd1.ID)

	assert.Error(t, err, "GetServiceDependencyByID shouldn't succeed after delete")
}
