package main

import (
	"flag"
	"fmt"
	"log"
	"strings"
	"sync"

	"code.justin.tv/availability/goracle/statusserver/cataloghandlers"
)

var apiurl = "http://localhost:8000/api/v1"
var lat = false
var acct = false
var fsck = false
var dry = false
var debug = false

var latencyPercentiles = []int{50, 90, 99}

func debugf(format string, v ...interface{}) {
	if debug {
		log.Printf(format, v...)
	}
}
func addlat(c cataloghandlers.Component) error {
	//   "beanstalk:twitch-video-aws:ulcer:us-west-2"
	sn := strings.Split(*c.Label, ":")
	if len(sn) < 2 {
		return fmt.Errorf("Non-standard component label: %s", *c.Label)
	}
	var query string
	queries := make([]uint, len(latencyPercentiles))
	for i, p := range latencyPercentiles {
		switch *c.Type {
		case "alb":
			if len(sn) != 4 {
				return fmt.Errorf("Non-standard alb component label for: %s", *c.Label)
			}
			account, name, region := sn[1], sn[2], sn[3]
			query = fmt.Sprintf("cloudwatch.%s.%s.alb.%s.TargetResponseTime_p%d", account, region, name, p)
		case "elb":
			if len(sn) != 4 {
				return fmt.Errorf("Non-standard elb component label for: %s", *c.Label)
			}
			account, name, region := sn[1], sn[2], sn[3]
			query = fmt.Sprintf("cloudwatch.%s.%s.elb.%s.Latency_p%d", account, region, name, p)
		case "beanstalk":
			if len(sn) != 4 {
				return fmt.Errorf("Non-standard beanstalk component label for: %s", *c.Label)
			}
			account, name, region := sn[1], sn[2], sn[3]
			query = fmt.Sprintf("maxSeries(cloudwatch.%s.%s.beanstalk.%s.*.Latency_p%d)", account, region, name, p)
		case "visageClient":
			if len(sn) != 2 {
				return fmt.Errorf("Non-standard visage component label for: %s", *c.Label)
			}
			name := sn[1]
			query = fmt.Sprintf("maxSeries(scale(stats.timers.visage.production.all.hystrix.%s.*.totalDuration.upper_%d, 0.001))", name, p)
		default:
			return fmt.Errorf("Unkown component type %s for %s", *c.Type, *c.Name)
		}
		qtype := fmt.Sprintf("percentile_%d", p)
		atype := "max"
		latq := &cataloghandlers.Query{
			Type:          &qtype,
			Query:         &query,
			AggregateType: &atype,
		}
		if dry {
			log.Printf("would create query\n")
			continue
		}
		url := fmt.Sprintf("%s/queries/", apiurl)
		err := postjson(url, latq, latq)
		if err != nil {
			return err
		}
		queries[i] = *latq.ID
	}

	metname := *c.Name + ":latency"
	threshold := 5.0
	ctype := "latency"
	rollup := true
	autogen := true
	latquery := "percentile_90"

	latm := &cataloghandlers.Metric{
		Label:           &metname,
		Name:            &metname,
		Description:     &metname,
		AutoGenerated:   &autogen,
		Queries:         queries,
		Threshold:       &threshold,
		CalculationType: &ctype,
		ComponentRollup: &rollup,
		LatencyQuery:    &latquery,
	}
	if dry {
		log.Printf("would create metric and add to comp\n")
		return nil
	}
	url := fmt.Sprintf("%s/metrics/", apiurl)
	err := postjson(url, latm, latm)
	if err != nil {
		return err
	}

	newmets := append(*c.Metrics, *latm.ID)
	c.Metrics = &newmets
	url = fmt.Sprintf("%s/components/%d", apiurl, *c.ID)
	err = putjson(url, &c, &c)

	return nil
}

func ensurelat(c cataloghandlers.Component) error {
	for _, m := range *c.Metrics {
		url := fmt.Sprintf("%s/metrics/%d", apiurl, m)
		var met cataloghandlers.Metric
		err := getjson(url, &met)
		if err != nil {
			return err
		}
		if *met.LatencyQuery != "" {
			debugf("%s already has a latency_query", *c.Name)
			return nil
		}
	}
	return addlat(c)
}

var acctMap struct {
	m map[string]uint
	sync.Once
}

func dofsck(c cataloghandlers.Component) error {
	for _, m := range *c.Metrics {
		debugf("Checking component %d metric %d", c.ID, m)
		url := fmt.Sprintf("%s/metrics/%d", apiurl, m)
		var met cataloghandlers.Metric
		err := getjson(url, &met)
		if err != nil {
			return fmt.Errorf("Metric %d doesn't exist", m)
		}
		for _, q := range met.Queries {
			debugf("Checking component %d metric %d query %d", c.ID, m, q)
			url := fmt.Sprintf("%s/queries/%d", apiurl, q)
			var query cataloghandlers.Query
			err := getjson(url, &query)
			if err != nil {
				return fmt.Errorf("Query %d doesn't exist for metric %d", q, m)
			}

		}
	}
	return nil
}

func main() {
	var comps []cataloghandlers.Component
	flag.StringVar(&apiurl, "api", apiurl, "Base URL of the goracle catalog API")
	flag.BoolVar(&acct, "acct", acct, "fixup accounts")
	flag.BoolVar(&fsck, "fsck", fsck, "run consistency checks")
	flag.BoolVar(&lat, "lat", lat, "fixup latencies")
	flag.BoolVar(&dry, "dry", dry, "Dry run")
	flag.BoolVar(&dry, "n", dry, "Dry run")
	flag.BoolVar(&debug, "debug", debug, "Enable debug output")
	flag.BoolVar(&debug, "d", debug, "Enable debug output")
	flag.Parse()
	if !lat && !acct && !fsck {
		log.Fatal("need at least one action (-lat, -acct)")
	}
	if flag.NArg() > 0 {
		for _, a := range flag.Args() {
			var comp cataloghandlers.Component
			url := fmt.Sprintf("%s/components/%s", apiurl, a)
			err := getjson(url, &comp)
			if err != nil {
				log.Fatal(err)
			}
			comps = append(comps, comp)
		}
	} else {
		url := apiurl + "/components/"
		err := getjson(url, &comps)
		if err != nil {
			log.Fatal(err)
		}
	}
	for _, c := range comps {
		debugf("checking component: %s, %d, %s\n", *c.Name, *c.ID, *c.Type)
		if lat {
			err := ensurelat(c)
			if err != nil {
				log.Printf("Failed to ensure latencies for %s: %s", *c.Name, err)
			}
		}
		if fsck {
			err := dofsck(c)
			if err != nil {
				log.Printf("Found inconsistency in component %s: %s", *c.Name, err)
			}

		}
	}
}
