package main

import (
	"fmt"
	"os"

	"github.com/dsoo/rollrus"
	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
	c "code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/errorhandling"
	"code.justin.tv/availability/goracle/guardianauth"
	"code.justin.tv/availability/goracle/stats"
	"code.justin.tv/availability/goracle/statusserver"
)

func setupLogging() {
	rollrus.SetupLogging(c.Config.RollbarAccessToken, c.Config.RollbarEnvironment)
	loglevel, err := logrus.ParseLevel(c.Config.LogLevel)
	if err != nil {
		logrus.Fatal("Couldn't parse log level %s: %s", c.Config.LogLevel, err)
	}
	logrus.SetLevel(loglevel)
}

func setupConfig() {
	if len(os.Args) == 2 {
		if err := c.ParseConfig(os.Args[1]); err != nil {
			logrus.Fatalf("Error reading TOML config -- %s", err)
		}
	}

}

func main() {
	setupConfig()
	setupLogging()
	defer errorhandling.PanicHandler()

	var dsn string
	if c.Config.CatalogdbDSN != "" {
		dsn = c.Config.CatalogdbDSN
	} else if c.Config.CatalogdbEndpoint != "" {
		// TODO backwards compat, remove later
		dsn = fmt.Sprintf("mysql://goracle:%s@tcp(%s)/catalogdb?parseTime=True", c.Config.CatalogdbPassword, c.Config.CatalogdbEndpoint)
	} else {
		logrus.Fatal("No catalogDB database specified in configuration")
	}
	if c.Config.StatsdEnabled {
		logrus.Info("enabling stats collection")
		stats.EnableStats(&c.Config)
	}
	catalog.SetupDB(dsn)
	defer catalog.CloseDB()
	guardianauth.InitGuardianDB(c.OauthClientID(), c.OauthClientSecret(), dsn)

	statusserver.RunServer(c.AssetPath())

}
