package config

import (
	"os"
	"path"
	"fmt"

	"github.com/BurntSushi/toml"
	"github.com/sirupsen/logrus"
)

// Config holds global configuration settings
var Config Configuration

// Configuration holds config information for Goracle
type Configuration struct {
	assetPath            string `toml:"asset_path"`
	RollbarAccessToken   string `toml:"rollbar_access_token"`
	RollbarEnvironment   string `toml:"rollbar_environment"`
	CatalogdbPassword    string `toml:"catalogdb_password"`
	CatalogdbEndpoint    string `toml:"catalogdb_endpoint"`
	CatalogdbDSN         string `toml:"catalogdb_dsn"`
	OauthClientID        string `toml:"oauth_client_id"`
	OauthClientSecret    string `toml:"oauth_client_secret"`
	LogLevel             string `toml:"log_level"`
	HmsEskEndpoint       string `toml:"hms_esk_endpoint"`
	EnableGuardian       bool   `toml:"enable_guardian"`
	RequireHTTPS         bool   `toml:"require_https"`
	EnableGraphqlDevPage bool   `toml:"enable_graphql_dev_page"`
	ServerAddr           string `toml:"server_addr"`
	StatsdEnabled        bool   `toml:"statsd_enabled"`
	StatsdEndpoint       string `toml:"statsd_endpoint"`
	LDAPEndpoint         string `toml:"ldap_endpoint"`
	LDAPSSLVerify        bool   `toml:"ldap_ssl_verify"`
	EmailProtocol        string `toml:"email_protocol"`
	SMTPEndpoint         string `toml:"smtp_endpoint"`
	SlackToken           string `toml:"slack_token"`
	S2SServiceName       string `toml:"s2s_service_name"`
	EnableS2S            bool   `toml:"enable_s2s"`
	EnableGraphqlLog     bool   `toml:"enable_graphql_log"`
	CloudWatchBufferSize int    `toml:"cloudwatch_buffer_size"`
	CloudWatchRegion     string `toml:"cloudwatch_region"`
}

func init() {
	SetDefaultConfig()
}

// SetDefaultConfig sets up default config with sane defults for development this are(mostly) reset be config.toml in staging and production
func SetDefaultConfig() {
	Config.assetPath = "./assets/"
	Config.RollbarAccessToken = "a451e95ccb8842c0be920cd8e140886f"
	Config.RollbarEnvironment = "development"
	Config.CatalogdbPassword = ""
	Config.CatalogdbEndpoint = ""
	Config.CatalogdbDSN = ""
	Config.OauthClientID = "98af1670-f902-443c-9564-4a403694d3e3"
	Config.OauthClientSecret = "c2c78161f7acb0b01d9385a11cb7152e33ec6459"
	Config.LogLevel = logrus.InfoLevel.String()
	Config.HmsEskEndpoint = ""
	Config.EnableGuardian = true
	// Disable this until we implement health checks that ignore the HTTPS header
	Config.RequireHTTPS = true
	Config.EnableGraphqlDevPage = false
	Config.ServerAddr = ":8000"
	Config.StatsdEnabled = true
	Config.StatsdEndpoint = "statsd.internal.justin.tv:8125"
	Config.LDAPEndpoint = "ldap-usw2.internal.justin.tv:636"
	Config.LDAPSSLVerify = true
	Config.EmailProtocol = "smtp"
	Config.SMTPEndpoint = "smtp.internal.justin.tv:25"
	Config.SlackToken = ""
	Config.S2SServiceName = "service-catalog-dev"
	// This is necessary because in CI, our tests do not have creds to start S2S
	Config.EnableS2S = true
	Config.EnableGraphqlLog = false
	Config.CloudWatchBufferSize = 100000
	Config.CloudWatchRegion = "us-west-2"
}

// ParseConfig parses and exposes a global config struct
func ParseConfig(fn string) error {
	// Set
	// TODO: Parse command flags
	// TODO: Create defaults
	// Load config variables

	if _, err := toml.DecodeFile(fn, &Config); err != nil {
		return err
	}
	return nil
}

// AssetPath returns the path to the web assets
func AssetPath() string {
	return Config.assetPath
}

// GetEmailTemplatePath returns the path to the file p that should be in templates/email/.
func GetEmailTemplatePath(p string) string {
	return path.Join(Config.assetPath, path.Join("templates/email/", p))
}

// StaticConfigPath returns the path to a static config asset(the json files)
func StaticConfigPath(fn string) string {
	return path.Join(path.Join(Config.assetPath, "config"), fn)
}

func HMSEskEndpoint() string {
	env := os.Getenv("ENVIRONMENT")
	if env == "production" {
		return "https://hms-esk.internal.justin.tv/v1/"
	} else if env == "development" {
		return "https://hms-esk-development.internal.justin.tv/v1/"
	}
	return Config.HmsEskEndpoint
}

func S2sService() string {
	env := os.Getenv("ENVIRONMENT")
	if env == "production" {
		return "service-catalog-prod"
	} else if env == "development" {
		return "service-catalog-dev"
	}
	return Config.S2SServiceName
}


//Determine what the preferred host name is.
func ServiceCatalogHost() string {
	if IsProduction() {
		return "servicecatalog.internal.justin.tv"
	} else {
		return "servicecatalog-dev.internal.justin.tv"
	}
}

//Determine what the preferred host name for backend is.
func ServiceCatalogAPIHost() string {
	if IsProduction() {
		return "servicecatalog-api.internal.justin.tv"
	} else {
		return "servicecatalog-api-dev.internal.justin.tv"
	}
}

func ServiceCatalogURL() string {
	return fmt.Sprintf("https://%s", ServiceCatalogHost())
}

func ServiceCatalogAPIURL() string {
	return fmt.Sprintf("https://%s", ServiceCatalogAPIHost())
}

// Looks up the environment of the application
// via environment variables... yo dawg
func Environment() string {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		return "unknown"
	}
	return env
}

func IsProduction() bool {
	return Environment() == "production"
}

func GetGATrackingID() string {
	if IsProduction() {
		return "UA-97236991-1"
	} else {
		return "UA-105711273-1"
	}
}

// IsReadOnlyMode returns a boolean
// indicating if the backend is in readonly mode
func IsReadOnlyMode() bool {
	return os.Getenv("READ_ONLY_MODE") == "true"
}
