package config

import (
	"fmt"
	"os"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/sirupsen/logrus"
)

var sandstorm_arn_template string = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/goracle-%s"
var secret_prefix string = "goracle/goracle/%s/"

type secretCacheLine struct {
	secret string
	expiry time.Time
}

var secretCache map[string]secretCacheLine

func init() {
	secretCache = map[string]secretCacheLine{}
}

func getSecret(name string) (string, error) {
	// If the secret is in the cache and it hasn't expired yet, use it.
	cl, ok := secretCache[name]
	if ok {
		if cl.expiry.After(time.Now()) {
			return cl.secret, nil
		}
	}

	env := os.Getenv("ENVIRONMENT")

	// Config for us-west-2 for STS
	awsConfig := &aws.Config{Region: aws.String("us-west-2")}
	stsclient := sts.New(session.New(awsConfig))

	// Create stscreds.AssumeRoleProvider with a 15 minute duration
	// for the role <CROSS ACCOUNT ROLE FROM DASHBOARD HERE> in our main account
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      fmt.Sprintf(sandstorm_arn_template, env),
		Client:       stsclient,
	}

	// Create credentials and config using our Assumed Role that we
	// will use to access the main account with
	credentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(credentials)

	// Finally, create a secret manager that uses the cross-account
	// config.
	sandstormManager := manager.New(manager.Config{
		AWSConfig: awsConfig,
		TableName: "sandstorm-production",
		KeyID:     "alias/sandstorm-production",
	})

	secret, err := sandstormManager.Get(name)
	if err != nil {
		logrus.Warn("Unable to fetch sandstorm secret: ", name)
		logrus.Warn("Sandstorm error: ", err)
		return "", err
	}
	secretCache[name] = secretCacheLine{
		secret: string(secret.Plaintext[:]),
		expiry: time.Now().Add(1 * time.Minute),
	}
	return string(secret.Plaintext[:]), nil
}

// Returns the client ID used by guardian oauth
func OauthClientID() string {
	env := os.Getenv("ENVIRONMENT")
	oauthClientID := fmt.Sprintf(secret_prefix, env) + "oauth_client_id_noproxy"

	id, err := getSecret(oauthClientID)
	if err != nil {
		if Config.OauthClientID != "" {
			return Config.OauthClientID
		} else {
			panic(err)
		}
	}
	logrus.Info("Successfully fetched sandstorm secret:", oauthClientID)
	return id
}

// Returns the client secret used by guardian oauth
func OauthClientSecret() string {
	env := os.Getenv("ENVIRONMENT")
	oauthClientSecret := fmt.Sprintf(secret_prefix, env) + "oauth_client_secret_noproxy"

	secret, err := getSecret(oauthClientSecret)
	if err != nil {
		if Config.OauthClientSecret != "" {
			return Config.OauthClientSecret
		} else {
			panic(err)
		}
	}
	logrus.Debug("Successfully fetched sandstorm secret:", oauthClientSecret)
	return secret
}

func CatalogDBPassword() string {
	env := os.Getenv("ENVIRONMENT")
	catalogdbPasswordSecret := fmt.Sprintf(secret_prefix, env) + "catalogdb_password"
	password, err := getSecret(catalogdbPasswordSecret)
	if err != nil {
		if Config.CatalogdbPassword != "" {
			return Config.CatalogdbPassword
		} else {
			panic(err)
		}
	}
	return password
}

// func CatalogDBUsername() string {
// 	env := os.Getenv("ENVIRONMENT")
// 	catalogdbUsernameSecret := fmt.Sprintf(secret_prefix, env) + "catalogdb_username"
// 	username, err := getSecret(catalogdbUsernameSecret)
// 	if err != nil {
// 		if Config.CatalogdbUsername != "" {
// 			return Config.CatalogdbUsername
// 		} else {
// 			panic(err)
// 		}
// 	}
// 	return username
// }
