package externalclient

import (
	"bytes"
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/http"

	"code.justin.tv/availability/goracle/guardianauth"
	"encoding/base64"
	"github.com/sirupsen/logrus"
	"golang.org/x/oauth2"
	"fmt"
)

// GQLClientConfig is the config GQLClient
type GQLClientConfig struct {
	Endpoint string
	Username string
	Password string
}

type GqlClient struct {
	cfg         GQLClientConfig
	accessToken string
	logger      *logrus.Logger
	guardian    *guardianauth.GuardianClient
}

const oauthClientID = "d661144b-4cd8-4258-afa2-953ace1cf355"

// NewGqlClient will create a guardianClient with the proper endpoints
func NewGqlClient(cfg GQLClientConfig, logger *logrus.Logger) *GqlClient {
	return &GqlClient{
		cfg: cfg,
		guardian: &guardianauth.GuardianClient{
			Cfg: guardianauth.GuardianClientConfig{
				Username: cfg.Username,
				Password: cfg.Password,
				ClientID: oauthClientID,
			},
			Logger: logger,
		},
		logger: logger,
	}
}

func (c *GqlClient) SetAuthCookie(request *http.Request, token *oauth2.Token) error {
	tokenJSON, err := json.Marshal(token)
	if err != nil {
		return err
	}
	tokenStr := base64.StdEncoding.EncodeToString(tokenJSON)
	cAuth := http.Cookie{
		Name:    "scAuthToken",
		Value:   string(tokenStr),
		Path:    "/",
		Expires: token.Expiry,
	}
	request.AddCookie(&cAuth)
	cUser := http.Cookie{
		Name:    "scAuthUser",
		Value:   string(c.cfg.Username),
		Path:    "/",
		Expires: token.Expiry,
	}
	request.AddCookie(&cUser)
	return nil
}

// SendGQLQuery handles all authentication required to make a gql query to a service catalog guardian protected endpoint
func (c *GqlClient) SendGQLQuery(query string, variables interface{}, result interface{}) (err error) {
	body := struct {
		Query         string      `json:"query"`
		OperationName string      `json:"operationName"`
		Variables     interface{} `json:"variables"`
	}{
		Query:         query,
		OperationName: "",
		Variables:     variables,
	}

	bs := bytes.NewBuffer(nil)
	err = json.NewEncoder(bs).Encode(&body)
	if err != nil {
		return err
	}

	req, err := http.NewRequest("POST", c.cfg.Endpoint, bs)
	if err != nil {
		return err
	}

	accessToken, err := c.guardian.GetAccessToken()
	if err != nil {
		return err
	}

	token := &oauth2.Token{AccessToken: accessToken}
	token.SetAuthHeader(req)
	err = c.SetAuthCookie(req, token)
	if err != nil {
		return err
	}

	res, err := http.DefaultClient.Do(req)
	if err != nil {
		return err
	}

	fullBody, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return err
	}

	errorResult := struct {
		Errors []struct {
			Message string `json:"message"`
		} `json:"errors"`
	}{}

	err = json.NewDecoder(bytes.NewBuffer(fullBody)).Decode(&errorResult)
	if err != nil {
		return err
	}
	if len(errorResult.Errors) > 0 {
		err = errors.New("received error: " + errorResult.Errors[0].Message)
		return err
	}

	err = json.NewDecoder(bytes.NewBuffer(fullBody)).Decode(result)
	return err
}

// ServiceInput defines the structure of args to be passed to createService in service catalog
type ServiceInput struct {
	Name               string                    `json:"name"`
	Description        string                    `json:"description"`
	TeamID             string                    `json:"team_id"`
	PrimaryOwnerUID    string                    `json:"primary_owner_uid"`
	Type		string 	`json:"type"`
	PagerDuty          string                    `json:"pagerduty"`
	SlackChannelID     string                    `json:"slack_channel_id"` // You can get this by right clicking channel in slack ui
	Attributes         []*ServiceAttribute `json:"attributes"`
	ServiceUpstreams   []ServiceDependencyInput `json:"service_upstreams,omitempty"`
	ServiceDownstreams []ServiceDependencyInput `json:"service_downstreams,omitempty"`
}

// ServiceUpdateInput defines the structure of args to be passed to createService in service catalog
type ServiceUpdateInput struct {
	ID                 string  `json:"id"`
	Name               *string                    `json:"name, omitempty"`
	Description        *string                    `json:"description, omitempty"`
	TeamID             *string                    `json:"team_id, omitempty"`
	PrimaryOwnerUID    *string                    `json:"primary_owner_uid, omitempty"`
	Type		*string 	`json:"type, omitempty"`
	PagerDuty          *string                    `json:"pagerduty, omitempty"`
	SlackChannelID     *string                    `json:"slack_channel_id, omitempty"` // You can get this by right clicking channel in slack ui
	Attributes         *[]*ServiceAttribute `json:"attributes, omitempty"`
	ServiceUpstreams   *[]ServiceDependencyInput `json:"service_upstreams,omitempty"`
	ServiceDownstreams *[]ServiceDependencyInput `json:"service_downstreams,omitempty"`
}

// Attributes are kv style descriptors of services
type ServiceAttribute struct {
	Name  string  `json:"name"`
	Value string `json:"value"`
}

// ServiceDependencyInput define service dep graphs. They are always from root to downstream. e.g. Twilight root service, visage is downstream service
type ServiceDependencyInput struct {
	ID                  *string `json:"id"`
	RootServiceID       *string `json:"root_service_id"`
	DownstreamServiceID *string `json:"downstream_service_id"`
}

// ServiceOutput is used to unmarshal response from ServiceCatalog. It is connected to ServiceOutputRequest.
type ServiceOutput struct {
	ID           string        `json:"id"`
	Created      string        `json:"created"`
	Updated      string        `json:"updated"`
	Name         string        `json:"name"`
	Description  string        `json:"description"`
	Pagerduty    string        `json:"pagerduty"`
	TeamID       string        `json:"team_id"`
	PrimaryOwner *PrimaryOwner `json:"primary_owner"`
	Attributes   []*ServiceAttribute  `json:"attributes"`
	Type         string        `json:"type"`
}

// ServiceOutputRequest is used to request data from service catalog in a certain way. It must be kept up to date with
// Service Output.
const ServiceOutputString = `fragment serviceData on Service {
    id
    created
    updated
    name
    description
    pagerduty
    team_id
    primary_owner {
      cn
      uid
    }
    attributes {
      name
      value
    }
    type
  }`

// PrimaryOwner is the structure of the primary owner data returned.
type PrimaryOwner struct {
	CN  *string `json:"cn"`
	UID *string `json:"uid"`
}

// You can pass a service object to CreateService to create a new service. You will get a
func (c *GqlClient) CreateService(service *ServiceInput) (*ServiceOutput, error) {
	var createServiceQuery = `
  mutation CreateService($service: ServiceInput!) {
    service: createService(service: $service) {
      ...serviceData
    }
  }
  %s
`
	result := struct {
		Data struct {
			Service ServiceOutput `json:"service"`
		} `json:"data"`
		Errors interface{}
	}{}

	input := struct {
		Service ServiceInput `json:"service"`
	}{*service}

	createServiceQuery = fmt.Sprintf(createServiceQuery, ServiceOutputString)
	err := c.SendGQLQuery(createServiceQuery, input, &result)
	if err != nil {
		return nil, err
	}
	if result.Errors != nil {
		return nil, fmt.Errorf("Error encountered in query %v", result.Errors)
	}
	return &result.Data.Service, nil
}

func (c *GqlClient) UpdateService(service *ServiceUpdateInput) (*ServiceOutput, error) {
	var updateServiceQuery = `
  mutation UpdateServiceByID($id: ID!, $service: ServiceInput!) {
    service: updateService(id: $id, service: $service) {
      ...serviceData
    }
  }
  %s
`
	result := struct {
		Data struct {
			Service ServiceOutput `json:"service"`
		} `json:"data"`
		Errors interface{}
	}{}

	input := struct {
		Service ServiceUpdateInput `json:"service"`
		ID string `json:"id"`
	}{*service, service.ID}

	updateServiceQuery = fmt.Sprintf(updateServiceQuery, ServiceOutputString)
	err := c.SendGQLQuery(updateServiceQuery, input, &result)
	if err != nil {
		return nil, err
	}
	if result.Errors != nil {
		return nil, fmt.Errorf("Error encountered in query %v", result.Errors)
	}
	return &result.Data.Service, nil
}

// SimpleService is the structure of the SimpleGetService output.
type SimpleService struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

// SimpleGetServices shows you how to use SendGQLQuery to request all service names and ids in service catalog
func (c *GqlClient) SimpleGetServices() (services []SimpleService, err error) {
	result := struct {
		Data struct {
			Services []SimpleService `json:"services"`
		} `json:"data"`
		Errors interface{}
	}{}

	err = c.SendGQLQuery(
		`
          query Services {
            services {
              id
              name
            }
          }`,
		struct{}{},
		&result)
	if err != nil {
		return nil, err
	}

	if len(result.Data.Services) == 0 {
		err = errors.New("error creating service")
		return nil, err
	}

	services = result.Data.Services
	return services, nil
}
