package guardianauth

import (
	"sync"

	uuid "github.com/satori/go.uuid"
)

// localNonceBackend implements an in-memory nonce
type localNonceBackend struct {
	sync.RWMutex
	cache map[string]*nonceInfo
}

// NewLocalNonceBackend initializes a new in-memory nonce store
func NewLocalNonceBackend() nonceBackend {
	return &localNonceBackend{
		cache: map[string]*nonceInfo{},
	}
}

// storeNonceInfo inserts nonce info into the backend and returns the ID
// that it was stored under.
func (nb *localNonceBackend) storeNonceInfo(ni nonceInfo) (string, error) {
	nonce := uuid.NewV4().String()
	nb.Lock()
	defer nb.Unlock()

	nb.cache[nonce] = &ni
	return nonce, nil
}

// getAndClearNonceInfo retrieves and checks expiration of the nonce information.
func (nb *localNonceBackend) getAndClearNonceInfo(nonce string) (*nonceInfo, error) {
	// This function leaks nonces over time, but it's the best solution to
	// being able to track the redirect URI associated with the nonce.

	nb.RLock()
	defer nb.RUnlock()

	var ni *nonceInfo
	ni, valid := nb.cache[nonce]
	if !valid {
		return nil, nil
	}

	delete(nb.cache, nonce)

	return ni, nil
}
