package ldap

import (
	"crypto/tls"
	"crypto/x509"
	"fmt"
	"strconv"
	"strings"

	"github.com/sirupsen/logrus"
	"gopkg.in/ldap.v2"

	"code.justin.tv/availability/goracle/config"
)

type Client struct {
	conn LDAPSearchable
}

// Create a simple interface for testing purposes
// the underlying library's LDAP client implements this
// interface already.
type LDAPSearchable interface {
	Search(*ldap.SearchRequest) (*ldap.SearchResult, error)
	Close()
}

type UserInfo struct {
	EmployeeNumber uint32
	CN             string
	UID            string
	AmazonUID      string
	Department     string
	Manager        string
	BuildingName   string
}

var fieldsToFetch []string = []string{"cn", "uid", "twAmazonUID", "employeeNumber", "department", "manager", "buildingName"}

// NewLDAPClient returns an LDAP connection
// that is connected to the Twitch internal LDAP
// server, ready to issue searches
// The connection is anonymous and read only
// The client uses the singleton config to generate a sane
// client with proper options
func NewClient() (*Client, error) {
	tlsConfig := &tls.Config{
		InsecureSkipVerify: config.Config.LDAPSSLVerify,
		ServerName:         strings.Split(config.Config.LDAPEndpoint, ":")[0],
		RootCAs:            x509.NewCertPool(),
	}

	c, err := ldap.DialTLS("tcp", config.Config.LDAPEndpoint, tlsConfig)
	if err != nil {
		return nil, err
	}
	return &Client{
		conn: c,
	}, nil
}

// Allow overwriting the underlying connection (useful for tests)
func (c *Client) WithConnection(conn LDAPSearchable) *Client {
	c.conn = conn
	return c
}

// GetUserInfo takes a user employeeNumber and looks up the user,
// returning the common name (cn), and username (uid)
// Returns nil,nil if the user is not found, and an nil,err if an error occurs
func (c *Client) GetUserInfo(employeeNumber uint32) (*UserInfo, error) {
	// Craft the search request
	searchRequest := ldap.NewSearchRequest(
		"ou=Users,dc=justin,dc=tv",
		ldap.ScopeWholeSubtree, ldap.NeverDerefAliases, 0, 0,
		false,
		fmt.Sprintf("(&(objectClass=person)(employeeNumber=%d))", employeeNumber),
		fieldsToFetch,
		nil,
	)
	// Do the search
	results, err := c.conn.Search(searchRequest)
	if err != nil {
		return nil, err
	}
	// if there are no entries in the result, the user wasn't found
	// so return nil
	if len(results.Entries) == 0 {
		return nil, nil
	} else if len(results.Entries) > 1 {
		// This shouldn't ever happen...
		return nil, fmt.Errorf("multiple users with employeeNumber '%d' found", employeeNumber)
	}
	// Extract the desired attribute values
	userInfo, err := extractUserInfo(results.Entries[0])
	// Check for errors, and also check if the returned employeeNumber matches what we asked for
	if err != nil {
		return nil, err
	} else if userInfo.EmployeeNumber != employeeNumber {
		return nil, fmt.Errorf("employee number in query result does not match with request")
	}
	return userInfo, nil
}

// GetAllUserInfo returns information for every user in LDAP
// If any individual user causes an error, an error will be logged
// and the user will be omitted from the results
func (c *Client) GetAllUserInfo() ([]*UserInfo, error) {
	// Craft the search request
	searchRequest := ldap.NewSearchRequest(
		"ou=Users,dc=justin,dc=tv",
		ldap.ScopeWholeSubtree, ldap.NeverDerefAliases, 0, 0,
		false,
		fmt.Sprintf("(&(objectClass=person)(employeeNumber=*))"),
		fieldsToFetch,
		nil,
	)
	// Do the search
	results, err := c.conn.Search(searchRequest)
	if err != nil {
		return nil, err
	}
	// iterate over the resulting entries, transform them into UserInfo's
	// and return the UserInfo's
	userInfos := make([]*UserInfo, 0)
	for _, entry := range results.Entries {
		ui, err := extractUserInfo(entry)
		if err != nil {
			logrus.Warnf("could not extract UserInfo fields for LDAP DN '%s'", entry.DN)
			continue
		}
		userInfos = append(userInfos, ui)
	}
	return userInfos, nil
}

// looks for the cn and uid attributes and returns them
func extractUserInfo(data *ldap.Entry) (*UserInfo, error) {
	var cn string
	var uid string
	var employeeNumber uint64 // will convert to uint32 later
	var department string
	var manager string
	var buildingName string
	var amazonUID string
	var err error
	for _, attr := range data.Attributes {
		switch attr.Name {
		case "cn":
			// Make sure the cn information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple CN's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no CN found for DN '%s'", data.DN)
			}
			cn = attr.Values[0]
		case "uid":
			// Make sure the uid information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple UID's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no UID found for DN '%s'", data.DN)
			}
			uid = attr.Values[0]
		case "twAmazonUID":
			// Make sure the uid information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple AmzUID's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no AmzUID found for DN '%s'", data.DN)
			}
			amazonUID = attr.Values[0]
		case "employeeNumber":
			// Make sure the employeeNumber information is legit
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple EmployeeNumber's found for DN '%s'", data.DN)
			} else if len(attr.Values) == 0 {
				return nil, fmt.Errorf("no EmployeeNumber found for DN '%s'", data.DN)
			}
			employeeNumber, err = strconv.ParseUint(attr.Values[0], 10, 32)
			if err != nil {
				return nil, fmt.Errorf("could not parse employeeNumber: %s", err.Error())
			}
			if employeeNumber == 0 {
				return nil, fmt.Errorf("employee number cannot be 0")
			}
		case "department":
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple department's found for DN %s", data.DN)
			}
			department = attr.Values[0]
		case "manager":
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple manager`s found for DN '%s'", data.DN)
			}
			manager = attr.Values[0]
		case "buildingName":
			if len(attr.Values) > 1 {
				return nil, fmt.Errorf("multiple building's found for DN '%s'", data.DN)
			}
			buildingName = attr.Values[0]
		}
	}
	// Verify that we found all the required fields before just blindly returning
	// This should never be triggered, because the response _should_ include
	// the attribute and just have len(Attribute.Values) == 0
	// Defensive programming!
	if cn == "" {
		return nil, fmt.Errorf("no CN found for DN '%s'", data.DN)
	}
	if uid == "" {
		return nil, fmt.Errorf("no UID found for DN '%s'", data.DN)
	}
	if employeeNumber == 0 {
		return nil, fmt.Errorf("no employeeNumber found for DN '%s'", data.DN)
	}

	ui := &UserInfo{
		EmployeeNumber: uint32(employeeNumber),
		CN:             cn,
		UID:            uid,
		Department:     department,
		Manager:        manager,
		BuildingName:   buildingName,
		AmazonUID:      amazonUID,
	}
	return ui, nil
}

// Close closes the underlying connection of the LDAP Client
func (c *Client) Close() {
	c.conn.Close()
}
