package ldap

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"gopkg.in/ldap.v2"
)

// LDAPMock is a simple mock implementation for the
// LDAP connection behind a Client struct
type LDAPMock struct {
	hardcodedResults *ldap.SearchResult
	hardcodedError   error
}

func (lm *LDAPMock) Search(*ldap.SearchRequest) (*ldap.SearchResult, error) {
	// ignore the search and return something else
	return lm.hardcodedResults, nil
}

// do nothing
func (lm *LDAPMock) Close() {
	return
}

func (lm *LDAPMock) SetResults(results *ldap.SearchResult) {
	lm.hardcodedResults = results
}

func (lm *LDAPMock) SetError(err error) {
	lm.hardcodedError = err
}

// createLDAPEntry will create LDAPEntries for testing utility.
// It has a default set of attributes, which can be overriden by override
// e.g. pass {"cn": "joe"} to change cn to joe
func createLDAPEntry(override map[string][]string) (*ldap.Entry, error) {
	ldapAttributes := map[string][]string{
		"cn":             {"Cool Dude"},
		"uid":            {"cdude"},
		"employeeNumber": {"123456"},
		"twAmazonUID":    {"dudec"},
		"department":     {"Pltf/Svcs_Ops Exc_OE Qual_1T"},
		"manager":        {"Top Dude"},
		"buildingName":   {"SFO18"},
	}
	for k, v := range override {
		if _, ok := ldapAttributes[k]; !ok {
			return nil, fmt.Errorf("Error: Attempt to override attribute not in default: <key %s, value %s>", k, v)
		}
		ldapAttributes[k] = v
	}
	dn := fmt.Sprintf("cn=%s,ou=Users,dc=justin,dc=tv", ldapAttributes["cn"])
	var attributes []*ldap.EntryAttribute
	for n, v := range ldapAttributes {
		attributes = append(attributes, &ldap.EntryAttribute{
			Name:   n,
			Values: v,
		})
	}
	ldapEntry := ldap.Entry{
		DN:         dn,
		Attributes: attributes,
	}
	return &ldapEntry, nil
}

func TestGetUserInfo(t *testing.T) {

	mockConn := &LDAPMock{}
	client := &Client{}
	client = client.WithConnection(mockConn)

	var mockResults *ldap.SearchResult
	var userInfo *UserInfo
	var err error

	// Happy path case
	ldapEntry, err := createLDAPEntry(make(map[string][]string))
	assert.Nil(t, err)
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{ldapEntry},
	}
	mockConn.SetResults(mockResults)
	userInfo, err = client.GetUserInfo(123456)
	assert.Nil(t, err)
	assert.Equal(t, "Cool Dude", userInfo.CN)
	assert.Equal(t, "cdude", userInfo.UID)
	assert.Equal(t, "dudec", userInfo.AmazonUID)
	assert.Equal(t, uint32(123456), userInfo.EmployeeNumber)
	assert.Equal(t, "Pltf/Svcs_Ops Exc_OE Qual_1T", userInfo.Department)
	assert.Equal(t, "Top Dude", userInfo.Manager)
	assert.Equal(t, "SFO18", userInfo.BuildingName)

	// Employee number doesn't match the returned result
	ldapEntry, err = createLDAPEntry(map[string][]string{
		"employeeNumber": {"666"}})
	assert.Nil(t, err)
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{ldapEntry},
	}
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{
			ldapEntry,
		},
	}
	mockConn.SetResults(mockResults)
	userInfo, err = client.GetUserInfo(123456)
	assert.NotNil(t, err)

	// Employee number isn't a parsable uint32
	ldapEntry, err = createLDAPEntry(map[string][]string{
		"employeeNumber": {"uwuwuwuwuwu"}})
	assert.Nil(t, err)
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{
			ldapEntry,
		},
	}
	mockConn.SetResults(mockResults)
	userInfo, err = client.GetUserInfo(123456)
	assert.NotNil(t, err)

	// User not found
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{},
	}
	mockConn.SetResults(mockResults)
	userInfo, err = client.GetUserInfo(123456)
	assert.Nil(t, err)
	assert.Nil(t, userInfo)

	// Multiple Users found for single lookup
	ldapEntry, err = createLDAPEntry(map[string][]string{})
	assert.Nil(t, err)
	ldapEntry2, err := createLDAPEntry(map[string][]string{
		"cn":             {"Uncool Person"},
		"uid":            {"uid"},
		"employeeNumber": {"12345"},
	})
	assert.Nil(t, err)
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{
			ldapEntry,
			ldapEntry2,
		},
	}
	mockConn.SetResults(mockResults)
	userInfo, err = client.GetUserInfo(123456)
	assert.NotNil(t, err)
}

func TestExtractUserInfo(t *testing.T) {

	var userInfo *UserInfo
	var err error
	var entry *ldap.Entry

	// Happy path
	entry, err = createLDAPEntry(map[string][]string{})
	assert.Nil(t, err)

	userInfo, err = extractUserInfo(entry)
	assert.Nil(t, err)
	assert.Equal(t, "Cool Dude", userInfo.CN)
	assert.Equal(t, "cdude", userInfo.UID)
	assert.Equal(t, "dudec", userInfo.AmazonUID)
	assert.Equal(t, uint32(123456), userInfo.EmployeeNumber)
	assert.Equal(t, "Pltf/Svcs_Ops Exc_OE Qual_1T", userInfo.Department)
	assert.Equal(t, "Top Dude", userInfo.Manager)
	assert.Equal(t, "SFO18", userInfo.BuildingName)

	// No CN for a user
	entry = &ldap.Entry{
		DN: "cn=Foobar Garply,ou=Users,dc=justin,dc=tv",
		Attributes: []*ldap.EntryAttribute{
			{
				Name:   "uid",
				Values: []string{"fgarply"},
			},
			{
				Name:   "employeeNumber",
				Values: []string{"654321"},
			},
		},
	}
	userInfo, err = extractUserInfo(entry)
	assert.NotNil(t, err)

	// No UID for a user
	entry = &ldap.Entry{
		DN: "cn=Foobar Garply,ou=Users,dc=justin,dc=tv",
		Attributes: []*ldap.EntryAttribute{
			{
				Name:   "cn",
				Values: []string{"Foobar Garply"},
			},
			{
				Name:   "employeeNumber",
				Values: []string{"654321"},
			},
		},
	}
	userInfo, err = extractUserInfo(entry)
	assert.NotNil(t, err)

	// No EmployeeNumber for a user
	entry = &ldap.Entry{
		DN: "cn=Foobar Garply,ou=Users,dc=justin,dc=tv",
		Attributes: []*ldap.EntryAttribute{
			{
				Name:   "cn",
				Values: []string{"Foobar Garply"},
			},
			{
				Name:   "uid",
				Values: []string{"fgarply"},
			},
		},
	}
	userInfo, err = extractUserInfo(entry)
	assert.NotNil(t, err)

	// Check Multiple error cases
	fieldsToCheck := []string{"cn", "uid", "twAmazonUID", "employeeNumber", "department", "manager", "buildingName"}
	for _, field := range fieldsToCheck {
		entry, err = createLDAPEntry(map[string][]string{
			field: {"Value1", "Value2"}})
		assert.Nil(t, err, fmt.Sprintf("Could not create entry for field %s", field))
		userInfo, err = extractUserInfo(entry)
		assert.NotNil(t, err, fmt.Sprintf("Did not flag error for duplicate of field %s", field))
	}

	// Check Empty Value error cases
	fieldsToCheck = []string{"cn", "uid", "twAmazonUID", "employeeNumber"}
	for _, field := range fieldsToCheck {
		entry, err = createLDAPEntry(map[string][]string{
			field: {}})
		assert.Nil(t, err, fmt.Sprintf("Could not create entry for field %s", field))
		userInfo, err = extractUserInfo(entry)
		assert.NotNil(t, err, fmt.Sprintf("Did not flag error for empty field %s", field))
	}
}

func TestGetAllUsers(t *testing.T) {

	mockConn := &LDAPMock{}
	client := &Client{}
	client = client.WithConnection(mockConn)

	var mockResults *ldap.SearchResult
	var userInfos []*UserInfo
	var err error

	// One big list with some good users, some bad
	// The bad ones shouldn't make it to the final list
	mockResults = &ldap.SearchResult{
		Entries: []*ldap.Entry{
			{
				DN: "cn=George Harrison,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"George Harrison"},
					},
					{
						Name:   "uid",
						Values: []string{"gharrison"},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"99999"},
					},
				},
			},
			{
				DN: "cn=Bad Dude,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"Cool Dude"},
					},
					{
						Name:   "uid",
						Values: []string{},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"22222"},
					},
				},
			},
			{
				DN: "cn=No UID,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"No UID"},
					},
					{
						Name:   "uid",
						Values: []string{},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"3333"},
					},
				},
			},
			{
				DN: "cn=John Lennon,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"John Lennon"},
					},
					{
						Name:   "uid",
						Values: []string{"sgt_pepper"},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"9"},
					},
				},
			},
			{
				DN: "cn=Bad EmpNumber,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"Bad EmpNumber"},
					},
					{
						Name:   "uid",
						Values: []string{"bademp"},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"1h2h3h4h5"},
					},
				},
			},
			{
				DN: "cn=Paul McCartney,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"Paul McCartney"},
					},
					{
						Name:   "uid",
						Values: []string{"pmccartney"},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"5555"},
					},
				},
			},
			{
				DN: "cn=Ringo Starr,ou=Users,dc=justin,dc=tv",
				Attributes: []*ldap.EntryAttribute{
					{
						Name:   "cn",
						Values: []string{"Ringo Starr"},
					},
					{
						Name:   "uid",
						Values: []string{"rstarr"},
					},
					{
						Name:   "employeeNumber",
						Values: []string{"1234"},
					},
				},
			},
		},
	}

	mockConn.SetResults(mockResults)
	userInfos, err = client.GetAllUserInfo()
	assert.Equal(t, 4, len(userInfos))
	assert.Nil(t, err)

}
