package messaging

import (
	"github.com/rs/cors"

	"code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/guardianauth"

	goji "goji.io"
	"goji.io/pat"
)

var teamIDs = [...]string {"anubhaws", "lukemng", "ajwlso", "mallouhm", "beardenj"}

type MessageInput struct {
	Body        string
	To          []string
	Subject     string
	ReplyTo     []string
	ServiceName string
	ServiceID   string
}

func RegisterHandlers(mux *goji.Mux) {
	c := cors.New(cors.Options{
		AllowedOrigins: []string{"http://localhost:3000", "http://servicecatalog-local.internal.justin.tv:3000", "https://servicecatalog-dev.internal.justin.tv", "https://servicecatalog.internal.justin.tv", "https://servicecatalog-api-dev.internal.justin.tv", "https://servicecatalog-api.internal.justin.tv"},
		AllowedMethods: []string{"GET", "POST", "HEAD", "OPTIONS", "PUT", "DELETE", "PATCH"},
		AllowedHeaders:   []string{"*"},
		MaxAge: 86400,
		AllowCredentials: true,
	})

	// Look at config to decide what kind of email handler to create
	if config.Config.EmailProtocol == "smtp" {
		emailClient = &SMTPEmailClient{
			MailEndpoint: config.Config.SMTPEndpoint,
		}
	} else if config.Config.EmailProtocol == "ses" {
		emailClient = &SESEmailClient{}
	}

	con := goji.SubMux()
	con.Use(c.Handler)
	con.Use(guardianauth.GuardianValidation)

	mux.Handle(pat.New("/messaging/*"), con)

	con.HandleFunc(pat.Post("/email/"), sendEmailHandler)
	con.HandleFunc(pat.Post("/slack/"), sendSlackHandler)
	con.HandleFunc(pat.Get("/serviceAudit/*"), slackValidationHandler)
	con.HandleFunc(pat.Get("/assignOwner/*"), slackAssignOwnerHandler)
}

func IsDevTeamMember(userID string) bool {
	for _, teamID := range teamIDs {
		if teamID == userID {
			return true
		}
	}
	return false
}
