import pg8000
from getpass import getpass
import json
import sys
import requests
import argparse
import csv

parser = argparse.ArgumentParser(description='Sync Org hierarchy data in the service catalog with RedShift upstream')
parser.add_argument(
    "--catalog-endpoint", default="http://localhost:8000/api/v2/query", type=str,dest="endpoint", help="catalog graphql endpoint"
)
parser.add_argument(
    "--db", default="capacity-prod.capxk4j4th11.us-west-2.redshift.amazonaws.com", dest="db", help="redshift db hostname"
)
parser.add_argument(
    "--no-associate-accounts", dest="associate_accounts", action="store_false", help="dont re-associate accounts to orgs"
)
parser.add_argument(
    "--no-associate-teams", dest="associate_teams", action="store_false", help="dont re-associate teams to orgs"
)
parser.add_argument(
    "--team-mapping", type=str, dest="team_mapping", help="path to CSV file with teamId,orgLabel map"
)
args = parser.parse_args()
# if we are associating teams, the team_mapping must be set
if args.associate_teams and not args.team_mapping:
    print("ERROR: Must specify '--team-mapping <CSV>' unless '--no-associate-teams' is set")
    sys.exit(1)

# Define some globals for the script based on the flags
GRAPHQL_ENDPOINT = args.endpoint
REDSHIFT_HOST = args.db
ASSOCIATE_TEAMS = args.associate_teams
ASSOCIATE_ACCOUNTS = args.associate_accounts
TEAM_ORG_MAP_PATH = args.team_mapping

# Top level org names
TOP_LEVEL_ORGS = ['TWITCH', 'BUS_OPERATIONS', 'GAME_COMMERCE', 'PLATFORM_SERVICE', 'CONSUMER_PRODUCT']


# Define some graphql queries and mutations that will be needed
UPDATE_ACCOUNT_MUTATION = '''
mutation UpdateAccountMutation($id:ID!, $account:AccountInput!) {
  updateAccount(id:$id, account:$account) {
    id
    org_id
    alias
  }
}
'''

GET_ORGS_QUERY = '''
query GetOrgs {
  orgs {
    id
    name
    label
    team_ids
    account_ids
  }
}
'''

GET_ORG_BY_LABEL_QUERY = '''
query GetOrgByLabel($label:String) {
  orgs(label:$label) {
    id
    name
    label
    parent_id
  }
}
'''

GET_ACCOUNT_BY_ALIAS_QUERY = '''
query GetAccountByAlias($alias:String) {
  accounts(alias:$alias) {
    id
    alias
    aws_account_id
    org_id
  }
}
'''

CREATE_ORG_MUTATION = '''
mutation CreateOrg($org:OrgInput!) {
  createOrg(org:$org) {
    id
    name
    label
  }
}
'''

ASSIGN_ORG_PARENT_MUTATION = '''
mutation AssignParentOrg($id:ID!, $org:OrgInput!) {
  updateOrg(id:$id, org:$org) {
    id
    name
    label
    parent_id
  }
}
'''

GET_TEAMS_QUERY = '''
query GetTeams {
  teams {
    id
    name
    org_id
  }
}
'''

UPDATE_TEAM_MUTATION = '''
mutation UpdateTeamMutation($id:ID!, $team:TeamInput!) {
  updateTeam(id:$id, team:$team) {
    id
    name
    org_id
  }
}
'''

DELETE_ORG_MUTATION = '''
mutation DeleteOrg($id:ID!) {
  deleteOrg(id:$id) {
    id
    name
    label
  }
}
'''

def create_db_conn():
    host, port, db_name = get_db_info()
    c = pg8000.connect(
        database=db_name,
        host=host,
        port=port,
        user='availability',
        password=getpass('Password for RedShift database connection: '),
        ssl=True
    )
    return c

def graphql(query, variables):
    resp = requests.post(GRAPHQL_ENDPOINT, data=json.dumps({"query": query, "variables": variables}))
    resp.raise_for_status()
    d = json.loads(resp.text)
    if "errors" in d:
        # Join all the errors and bail
        raise ValueError("ERROR: %s" % ";".join(map(lambda err: err['message'], d['errors'])))
    return d['data']

# Returns the db connection info in the following tuple:
# (host, port, db_name)
def get_db_info():
    return (REDSHIFT_HOST, 5439, "capacity")

    (host, port, db_name) = db.get_db_info()

# Helper function to run a redshift query and return the results
def get_query_results(conn, query):
    curr = conn.cursor()
    curr.execute(query)
    return curr.fetchall()

def get_org_from_fleet(fleet_to_org_map, fqfn):
    fqfn_fragments = fqfn.split('/')
    for i in range(len(fqfn_fragments), 0, -1):
        key = '/'.join(fqfn_fragments[:i])
        lookup = fleet_to_org_map.get(key, None)
        if lookup:
            return lookup
    raise KeyError('No fqfn in org-fleet map: ' + fqfn)

# Returns a list of dicts with the following keys
# org_label: unique label for an org
# aws_account_id: an AWS account ID that should be associated to the org label
# aws_account_alias: an AWS account alias for the aws_account_id
def get_org_account_relationships(c):
    # Use pg8000 to connect to Redshift
   
    org_records = get_query_results(c, 'select name,id,parent_id,owner_name from twitch_org')
    fleet_mapping_records = get_query_results(c, 'select owner_id,fqfn from twitch_aws_fleet_mapping')
    prod_account_records = get_query_results(c, 'select account_id,alias from aws_prod_account_list')
    fleet_account_records = get_query_results(c, 'select account_id,fqfn,_timestamp from fleet_accounts')

    # UTILITY DATA STRUCTURES
    # List of accounts id strings as a flat array
    prod_account_ids = map(lambda acct_data: acct_data[0], prod_account_records)
    # Mapping from fqfn to owner org
    fleet_to_org_map = {}
    for org,fqfn in fleet_mapping_records:
        fleet_to_org_map[fqfn] = org
    # Mapping from prod account_id to account alias
    account_id_to_alias_map = {}
    for account_id,alias in prod_account_records:
        account_id_to_alias_map[account_id] = alias
    # Map org to it's parent
    org_to_parent_map = {}
    for org in org_records:
        org_to_parent_map[org[1]] = org[2]
    # END UTILITY DATA STRUCTURES


    # Drop fleet accounts that we dont care about (they aren't prod)
    fleet_account_records = filter(lambda fa: fa[0] in prod_account_ids, fleet_account_records)

    # Only keep the highest timestamp for each account
    # Create mapping of highest timestamps per account
    latest_timestamps = {}
    for account in fleet_account_records:
        account_id = account[0]
        timestamp = int(account[2])
        if account_id in latest_timestamps:
            if timestamp > latest_timestamps[account_id]:
                latest_timestamps[account_id] = timestamp
        else:
            latest_timestamps[account_id] = timestamp
    # Filter fleet_accounts on highest timestamp equality
    fleet_account_records = filter(lambda fa: fa[2] == latest_timestamps[fa[0]], fleet_account_records)

    # Construct final record set
    data = []
    for fleet_account in fleet_account_records:
        fqfn = fleet_account[1]
        try:
            org = get_org_from_fleet(fleet_to_org_map, fqfn)
        except:
            print("Error processing fleet_account with fqfn: {fqfn}... Skipping".format(fqfn=fqfn))
            continue
        # Bubble the org up to a top-level org
        while org not in TOP_LEVEL_ORGS:
            org = org_to_parent_map[org]
        account_id = fleet_account[0]
        alias = account_id_to_alias_map[account_id]
        data.append({
            'org_label': org,
            'aws_account_id': account_id,
            'aws_account_alias': alias
        })

    return data

def get_real_org_data(c):
    org_records = get_query_results(c, "select name,id,parent_id from twitch_org where id = 'TWITCH' or parent_id = 'TWITCH'")
    org_mapper = lambda org: {
                    "name": org[0],
                    "label": org[1],
                    "parent_label": org[2]
                 }
    return map(org_mapper, org_records)

def get_catalog_org_data():
    return graphql(GET_ORGS_QUERY, {})['orgs']

def sync_orgs(real_orgs, catalog_orgs):

    orgs_to_create = []
    orgs_to_delete = []

    # recursive helper to sync orgs cleanly
    def sync_orgs_helper(real_orgs, catalog_orgs):
        # base case, we are done!
        if len(real_orgs) == 0 and len(catalog_orgs) == 0:
            return
        # need to delete the leftovers
        elif len(real_orgs) == 0:
            orgs_to_delete.append(catalog_orgs[0])
            return sync_orgs_helper(real_orgs, catalog_orgs[1:])
        # need to create the leftovers
        elif len(catalog_orgs) == 0:
            orgs_to_create.append(real_orgs[0])
            return sync_orgs_helper(real_orgs[1:], catalog_orgs)
        # need to delete this catalog org
        elif catalog_orgs[0]['label'] < real_orgs[0]['label']:
            orgs_to_delete.append(catalog_orgs[0])
            return sync_orgs_helper(real_orgs, catalog_orgs[1:])
        # need to create this org in the catalog
        elif catalog_orgs[0]['label'] > real_orgs[0]['label']:
            orgs_to_create.append(real_orgs[0])
            return sync_orgs_helper(real_orgs[1:], catalog_orgs)
        # the orgs match, so continue through both lists
        elif catalog_orgs[0]['label'] == real_orgs[0]['label']:
            # TODO: CHECK FOR NEEDED UPDATES TO 'name' and maintain an 'orgs_to_update' list
            return sync_orgs_helper(real_orgs[1:], catalog_orgs[1:])

    # This will fill out orgs_to_create and orgs_to_delete
    l = lambda o: o['label']
    # Sort on the label field so we can do a zipper compare with recursion
    sync_orgs_helper(sorted(real_orgs, key=l), sorted(catalog_orgs, key=l))

    # Delete the old orgs first to avoid org name conflicts
    for org in orgs_to_delete:
        print("deleting org with label '%s'" % org['label'])
        # TODO: REMOVE ASSOCIATIONS FROM DANGLING TEAMS AND ACCOUNTS
        graphql(DELETE_ORG_MUTATION, {'id': org['id']})
        # Remove dangling associations
        for team_id in org['team_ids']:
            graphql(UPDATE_TEAM_MUTATION, {'id': team_id, 'team': {'org_id': '0'}})
        for account_id in org['account_ids']:
            graphql(UPDATE_ACCOUNT_MUTATION, {'id': account_id, 'account': {'org_id': '0'}})
    # Go through and create the orgs
    for org in orgs_to_create:
        print("creating org with label '%s'" % org['label'])
        graphql(CREATE_ORG_MUTATION, {'org': {'name':org['name'],'label':org['label']}})
    # Assign each org to it's parent
    for org in real_orgs:
        assign_org_parent(org['label'], org['parent_label'])

    return

def assign_org_parent(child_label, parent_label):
    # Best effort attempt to assign the org to its proper parent
    if child_label == 'TWITCH':
        return
    try:
        # Get the org parent via the label
        child_org = graphql(GET_ORG_BY_LABEL_QUERY, {'label': child_label})['orgs']
        parent_org = graphql(GET_ORG_BY_LABEL_QUERY, {'label': parent_label})['orgs']
        if len(parent_org) != 1:
            raise ValueError(
                "did not find exactly one org to associate to '{}' as a parent! found {} orgs with label {}".format(
                    child_label, len(parent_org), parent_label
                )
            )
        if len(child_org) != 1:
            raise ValueError(
                "did not find exactly one org to associate '{}' as the parent for! found {} orgs with label {}".format(
                    parent_label, len(child_org), child_label
                )
            )
        # Turn list response into single item
        parent_org = parent_org[0]
        child_org = child_org[0]

        # Make the association if needed
        if str(child_org['parent_id']) != parent_org['id']:
            print("assigning the parent org of '%s' to '%s'" % (child_org['label'], parent_org['label']))
            graphql(ASSIGN_ORG_PARENT_MUTATION, {'id': child_org['id'], 'org': {'parent_id': parent_org['id']}})

    except Exception as e:
        print(e)
        print("skipping org parent assignment for org with label '%s'" % child_label)

def assign_account_to_org(acct_org_data):
    try:
        # Get the org
        org = graphql(GET_ORG_BY_LABEL_QUERY, {'label': acct_org_data['org_label']})['orgs']
        org = org[0]

        # Get the account
        account = graphql(GET_ACCOUNT_BY_ALIAS_QUERY, {'alias': acct_org_data['aws_account_alias']})['accounts']
        if len(account) == 0:
            print("account not found in catalog '%s'" % acct_org_data['aws_account_alias'])
            return
        account = account[0]

        # Make the association
        if account['org_id'] != org['id']:
            print("assigning AWS account with alias '%s' to org with label '%s'" % (account['alias'], org['label']))
            graphql(UPDATE_ACCOUNT_MUTATION, {'id': account['id'], 'account': {'org_id': org['id']}})
    except Exception as e:
        print("ERROR: %s" % str(e))
        print("skipping account to org association for AWS account with alias '%s'" % acct_org_data['aws_account_alias'])

def associate_teams():
        with open(TEAM_ORG_MAP_PATH, 'r') as csvfile:
            reader = csv.reader(csvfile, delimiter=',')
            for row in reader:
                assign_team_to_org(row[0], row[1])

def assign_team_to_org(team_id, org_label):
    try:
        org = graphql(GET_ORG_BY_LABEL_QUERY, {'label': org_label})['orgs']
        org = org[0]
        graphql(UPDATE_TEAM_MUTATION, {'id': team_id, 'team': {'org_id': org['id']}})
    except Exception as e:
        print("ERROR: %s" % str(e))
        print("skipping team to org association for team with id '%s'" % team_id)


def main():
    c = create_db_conn()
    
    # ORGS
    # Get the orgs as seen by redshift (legit) and catalog (possibly out of date)
    print("updating orgs...")
    real_orgs = get_real_org_data(c)
    catalog_orgs = get_catalog_org_data()
    # Compare the two sets of orgs, bringing the catalog
    # in sync with redshift. This will add redshift orgs not in the catalog,
    # and delete catalog orgs that are no longer in redshift.
    sync_orgs(real_orgs, catalog_orgs)
    print("done!\n")

    # TEAMS
    # If given a team id->org alias CSV mapping, attempt to re-link all
    # teams in the CSV
    # TODO: do some file loading, flag checking, etc
    if ASSOCIATE_TEAMS:
        print("updating teams according to %s..." % TEAM_ORG_MAP_PATH)
        associate_teams()
        print("done!\n")

    # ACCOUNTS
    # Associate AWS Accounts to the proper org
    if ASSOCIATE_ACCOUNTS:
        print("updating AWS accounts...")
        account_orgs = get_org_account_relationships(c)
        for acct in account_orgs:
            assign_account_to_org(acct)
        print("done!")


if __name__ == "__main__":
    main()

# General steps:
# 0. Get list of all existing orgs in catalog
# 1. Get list of real orgs in RedShift
# 2. Create newly detected orgs
# 3. Delete Removed orgs
#    * Unassociate teams to that org
#    * Unassociate aws accounts to that org
# 4. Run through RedShift AWS Account Mapping logic
# 5. Check for team mapping CSV, teamId -> org alias, do associations in the file


