#! /bin/bash
set -u

# Lint, test, and generate coverage.

# Checks all child packages by default or the list of packages
# specified on the command line

# proto and mocks packages are skipped because they are generated sources.

WARN=0
FATAL=0
OUT=${OUT:-.}

function warn() {
    (( WARN++ ))
    echo "+++ WARNING detected"
}

function fatal() {
    (( FATAL++ ))
    echo "*** FATAL error detected"
}

echo "===================== Linting and testing Go Code ======================"
# use packages from command line or find child packages
PKGS=${@:-$(go list ./... | grep -v vendor)}
# set flag to enable v1 api for testing
export ENABLE_V1_API=true
for pkg in $PKGS; do
  echo "Linting: ${pkg}"
  golint ${pkg} | grep . && warn

  echo "Static error analysis: ${pkg}"
  go vet "${pkg}" || warn
  errcheck -blank "${pkg}" ||

  echo "Testing and generating coverage for: ${pkg}"
  coverage_prefix=${OUT}/${pkg//\//_}
  go test -coverprofile=${coverage_prefix}.out ${pkg} || fatal
  if [[ -f ${coverage_prefix}.out ]]; then
    go tool cover -func=${coverage_prefix}.out
    go tool cover -html=${coverage_prefix}.out -o ${coverage_prefix}.html
    cat ${coverage_prefix}.out >> coverage.txt
    rm -rf ${coverage_prefix}.out
  fi
  echo
done
echo
echo "$0: ${FATAL} fatal errors and ${WARN} warnings"
echo "========================================================================"
echo
