package serverutil

import (
	"encoding/json"
	"net/http"
	"strconv"
	"time"

	"github.com/sirupsen/logrus"
)

func StringToUint(in string) (uint, error) {
	out, err := strconv.ParseUint(in, 10, 64)
	if err != nil {
		return 0, err
	}
	return uint(out), nil
}

// RequireHTTPS checks if communication is over https, if not, redirect.
func RequireHTTPS(inner http.Handler) (outer http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		proto, found := r.Header["X-Forwarded-Proto"]
		if !found || proto[0] != "https" {
			http.Redirect(w, r, "https://"+r.Host+r.RequestURI, http.StatusSeeOther)
		}
		inner.ServeHTTP(w, r)
	})
}


// This allows us to capture status code for logging
type loggingResponseWriter struct {
	http.ResponseWriter
	statusCode int
}

func NewLoggingResponseWriter(w http.ResponseWriter) *loggingResponseWriter {
	return &loggingResponseWriter{w, http.StatusOK}
}

func (lrw *loggingResponseWriter) WriteHeader(code int) {
	lrw.statusCode = code
	lrw.ResponseWriter.WriteHeader(code)
}

// Does basic HTTP logging
func Logger(inner http.Handler) (outer http.Handler) {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		lrw := NewLoggingResponseWriter(w)
		inner.ServeHTTP(lrw, r)
		logrus.Info(
			r.Method, " ",
			r.RequestURI, " ",
			lrw.statusCode, " ",
			time.Since(start),
		)
	})
}

// APIErrorMsg is a convenience struct
// used for solely for marshaling to JSON
// to create a consistently formatted response
// in the event of API errors
type APIErrorMsg struct {
	Errors []string `json:"errors"`
}

// HandleAPIErrors will take a response writer, a list of errors,
// and a desired HTTP response code and will craft a response to the client
// containing well formatted error descriptions.
// Use this method for consistent error message formats across the API
func HandleAPIErrors(w http.ResponseWriter, errs []error, statusCode int) {
	msgs := make([]string, 0)
	for _, err := range errs {
		msgs = append(msgs, err.Error())
	}
	apiErr := APIErrorMsg{Errors: msgs}
	// Try to marshal the error message for response
	errMsg, err := json.Marshal(apiErr)
	if err != nil {
		// hopefully this never happens, but it could!
		statusCode = 500
		errMsg = []byte("{\"errors\": [\"internal server error\"]")
	}
	// We could probably check errors here, but we are already in error handling
	// land so why even bother, just try to get the client an error message
	// in a best effort fashion.
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(statusCode)
	w.Write(errMsg)
}
