package stats

// This file creates the statsd client that is used
// throughout this package to submit internal goracle
// stats to statsd

import (
	"fmt"
	"time"

	"code.justin.tv/availability/goracle/config"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"
	"code.justin.tv/commerce/splatter"
)

const (
	namespace = "goracle"
	twitchTelemetryFlush     = 30 * time.Second // Flush metrics every 30 seconds for TwitchTelemetry
	twitchTelemetryMinBuffer = 100000           // Define a min buffer size to ensure there is enough room for metrics while flushing
)

var statsdClient statsd.Statter

func init() {
	var err error
	// Get config vars
	// - statsd_enabled
	// - statsd_endpoint
	// - environment?
	statsdClient, err = statsd.NewNoopClient()
	if err != nil {
		logrus.Fatalf("failed to create statsd client: %s", err.Error())
	}
}

func setupTwitchTelemetryStatter(namespace string, deploymentStage string, goracleConfig *config.Configuration) statsd.Statter {
	// Define a buffer as at least twitchTelemetryMinBuffer
	bufferSize := goracleConfig.CloudWatchBufferSize
	if bufferSize < twitchTelemetryMinBuffer {
		bufferSize = twitchTelemetryMinBuffer
	}

	// Construct the TwitchTelemetry Config
	twitchTelemetryConfig := &splatter.BufferedTelemetryConfig{
		FlushPeriod:       twitchTelemetryFlush,
		BufferSize:        bufferSize,
		AggregationPeriod: time.Minute, // Place all metrics into 1 minute buckets
		ServiceName:       namespace,
		AWSRegion:         goracleConfig.CloudWatchRegion,
		Stage:             deploymentStage,
		Substage:          "primary", // Set to primary for now. If there's a canary substage in prod in the future, then use it
		Prefix:            "", // No need for a prefix since service, stage, and substage are all dimensions
	}

	// filtered map can be empty
	filteredMetrics := map[string]bool{}

	// Construct the buffered CloudWatch TwitchTelemetry sender
	return splatter.NewBufferedTelemetryCloudWatchStatter(twitchTelemetryConfig, filteredMetrics)
}

func EnableStats(goracleConfig *config.Configuration) error {
	var err error
	env := config.Environment()
	if env == "unknown" {
		logrus.Warn("refusing to enable statsd metrics when environment is unknown")
		return fmt.Errorf("cannot create statsd client without an environment set")
	}

	statsdClient = setupTwitchTelemetryStatter(namespace, env, goracleConfig)
	// We should bail hard if we can't create a statsd client (noop or otherwise)
	// for any reason
	if err != nil {
		msg := fmt.Sprintf("failed to create statsd client: %s", err.Error())
		logrus.Fatalf(msg)
		return fmt.Errorf(msg)
	}
	return nil
}

func DisableStats() error {
	var err error
	statsdClient, err = statsd.NewNoopClient()
	return err
}

func StatsdClient() statsd.Statter {
	return statsdClient
}
