package stats

import (
	"fmt"
	"time"

	"github.com/sirupsen/logrus"
)

// Submits a timing statistic for the GraphQL API
// Expects:
// - graphqlName - the name of the top level entry point into the graphql schema
// - graphqlType - either "query" or "mutation"
// - success - whether or not the request encountered an error
// - start, end - the start and end time of the request, used to create the timing value
func SubmitAPITiming(graphqlName, graphqlType string, success bool, duration time.Duration) {
	var err error

	eventString, eventStringTotal, err := formatStatStrings(graphqlName, graphqlType, success)
	if err != nil {
		// Log the error and swallow it
		logrus.Warn(err.Error())
		return
	}

	// Submit a timer for just the status (success/failure)
	err = statsdClient.TimingDuration(eventString, duration, 1)
	if err != nil {
		logrus.Warnf("error submitting timer '%s': %s", eventString, err.Error())
	}
	// Submit a timer for overall request timing tracking (error or otherwise)
	err = statsdClient.TimingDuration(eventStringTotal, duration, 1)
	if err != nil {
		logrus.Warnf("error submitting timer '%s': %s", eventStringTotal, err.Error())
	}
	// Submit a counter for the number of times this graphql point was hit
	// probably redundant with the timer, but lets just track it anyways, no biggie
	err = statsdClient.Inc(eventString, 1, 1)
	if err != nil {
		logrus.Warnf("error submitting counter '%s': %s", eventString, err.Error())
	}
}

func formatStatStrings(requestName, requestType string, success bool) (event, total string, err error) {
	var statusString string
	if success {
		statusString = "success"
	} else {
		statusString = "failure"
	}

	if requestName == "" {
		return "", "", fmt.Errorf("error formatting stat name: requestName cannot be an empty string")
	}

	if requestType != "queries" && requestType != "mutations" {
		errMsg := fmt.Sprintf("expected graphql statistic to be of type 'queries' or 'mutations', got: %s", requestType)
		logrus.Warnf(errMsg)
		return "", "", fmt.Errorf(errMsg)
	}
	event = fmt.Sprintf("api.%s.%s.%s", requestType, requestName, statusString)
	total = fmt.Sprintf("api.%s.%s.total", requestType, requestName)
	return event, total, nil
}
