package stats

import (
	"testing"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/assert"
)

func TestFormatStatString(t *testing.T) {
	DisableStats()
	var event string
	var total string
	var err error
	// Happy path
	event, total, err = formatStatStrings("foobar", "queries", true)
	assert.Equal(t, "api.queries.foobar.success", event)
	assert.Equal(t, "api.queries.foobar.total", total)
	assert.Nil(t, err)
	event, total, err = formatStatStrings("baz", "queries", false)
	assert.Equal(t, "api.queries.baz.failure", event)
	assert.Equal(t, "api.queries.baz.total", total)
	assert.Nil(t, err)
	event, total, err = formatStatStrings("garply", "mutations", true)
	assert.Equal(t, "api.mutations.garply.success", event)
	assert.Equal(t, "api.mutations.garply.total", total)
	assert.Nil(t, err)
	event, total, err = formatStatStrings("coolNewOperation", "mutations", false)
	assert.Equal(t, "api.mutations.coolNewOperation.failure", event)
	assert.Equal(t, "api.mutations.coolNewOperation.total", total)
	assert.Nil(t, err)
	// Sad path
	// bad request type
	event, total, err = formatStatStrings("foobar", "notathing", true)
	assert.Equal(t, event, "")
	assert.Equal(t, total, "")
	assert.NotNil(t, err)
	// name is empty
	event, total, err = formatStatStrings("", "queries", true)
	assert.Equal(t, event, "")
	assert.Equal(t, total, "")
	assert.NotNil(t, err)
}

func TestSubmitAPIStats(t *testing.T) {
	mockOutStatter()
	d, _ := time.ParseDuration("300ms")
	// Each call should fire 2 timers and a counter
	SubmitAPITiming("foobar", "queries", true, d)
	s := StatsdClient().(*mockStatter)
	assert.Equal(t, 1, s.NumCounterCalls("api.queries.foobar.success"))
	assert.Equal(t, 1, s.NumTimerCalls("api.queries.foobar.success"))
	assert.Equal(t, 1, s.NumTimerCalls("api.queries.foobar.total"))
	SubmitAPITiming("foobar", "queries", true, d)
	SubmitAPITiming("foobar", "queries", true, d)
	SubmitAPITiming("foobar", "queries", true, d)
	assert.Equal(t, 4, s.NumCounterCalls("api.queries.foobar.success"))
	assert.Equal(t, 4, s.NumTimerCalls("api.queries.foobar.success"))
	assert.Equal(t, 4, s.NumTimerCalls("api.queries.foobar.total"))
	// Sending a bad metric should do nothing
	SubmitAPITiming("foobar", "asdasdasdfail", true, d)
	assert.Equal(t, 0, s.NumCounterCalls("api.asdasdasdfail.foobar.success"))
	assert.Equal(t, 0, s.NumTimerCalls("api.asdasdasdfail.foobar.success"))
	assert.Equal(t, 0, s.NumTimerCalls("api.asdasdasdfail.foobar.total"))

}

// Helper struct to mock out a Statter and track the number of calls

func mockOutStatter() {
	statsdClient = &mockStatter{
		timerCallMap:   make(map[string]int),
		counterCallMap: make(map[string]int),
	}
}

type mockStatter struct {
	timerCallMap   map[string]int
	counterCallMap map[string]int
}

func (ms *mockStatter) Inc(s string, v int64, sr float32) error {
	ms.mapTick(ms.counterCallMap, s, 1)
	return nil
}

func (ms *mockStatter) TimingDuration(s string, t time.Duration, sr float32) error {
	ms.mapTick(ms.timerCallMap, s, 1)
	return nil
}

func (ms *mockStatter) mapTick(m map[string]int, s string, i int) int {
	if _, found := m[s]; !found {
		m[s] = i
	} else {
		m[s] += i
	}
	return m[s]
}

func (ms *mockStatter) NumCounterCalls(s string) int {
	return ms.mapTick(ms.counterCallMap, s, 0)
}
func (ms *mockStatter) NumTimerCalls(s string) int {
	return ms.mapTick(ms.timerCallMap, s, 0)
}

// These are part of the interface, but unused for mock purposes.
func (mockStatter) Dec(string, int64, float32) error        { return nil }
func (mockStatter) Gauge(string, int64, float32) error      { return nil }
func (mockStatter) GaugeDelta(string, int64, float32) error { return nil }
func (mockStatter) Timing(string, int64, float32) error     { return nil }
func (mockStatter) Set(string, string, float32) error       { return nil }
func (mockStatter) SetInt(string, int64, float32) error     { return nil }
func (mockStatter) Raw(string, string, float32) error       { return nil }
func (mockStatter) NewSubStatter(string) statsd.SubStatter  { return nil }
func (mockStatter) SetPrefix(string)                        {}
func (mockStatter) Close() error                            { return nil }
