package stats

// This file defines tracing middleware for graphql
// queries.  These tracers are embedded into the graphql
// schema at parsing time, and gets applied to all graphql
// requests automatically.

import (
	"context"
	"strings"
	"time"

	"github.com/neelance/graphql-go/errors"
	"github.com/neelance/graphql-go/introspection"
	"github.com/neelance/graphql-go/trace"
	"github.com/sirupsen/logrus"
)

type Tracer struct {
	Prefix   string
    LogQuery bool
}

func StatsTracer() *Tracer {
	return &Tracer{}
}

func (t Tracer) TraceQuery(ctx context.Context, queryString string, operationName string, variables map[string]interface{}, varTypes map[string]*introspection.Type) (context.Context, trace.TraceQueryFinishFunc) {

	// Get a start time for the request evaluation
	start := time.Now()

	// This function fires after the request has completed
	return ctx, func(errs []*errors.QueryError) {
		// determine if the GraphQL request was successful
		var success bool
		if len(errs) == 0 {
			success = true
		} else {
			success = false
		}
		// determine if the GraphQL request was a query or mutation
		var requestType string
		trimmedQuery := strings.TrimSpace(queryString)
		if strings.HasPrefix(trimmedQuery, "mutation") {
			requestType = "mutations"
		} else {
			requestType = "queries"
		}
		// Get execution time for the request
		delta := time.Since(start)

		logrus.Info("query operation: ", requestType, " ", operationName)
        if t.LogQuery {
            // Debug logging of the query, if needed
            logrus.Debug(trimmedQuery)
        }
		if operationName == "" {
			logrus.Debug("WARNING: This query is missing an operation name")
			operationName = "unnamed"
		}
		if t.Prefix != "" {
			operationName = t.Prefix + "." + operationName
		}
		// Fire statistic!
		SubmitAPITiming(operationName, requestType, success, delta)
	}
}

// At this time we don't track individual Field evaluation
func (t Tracer) TraceField(ctx context.Context, label, typeName, fieldName string, trivial bool, args map[string]interface{}) (context.Context, trace.TraceFieldFinishFunc) {
	return ctx, func(errs *errors.QueryError) {
		return
	}
}
