package statusserver

import (
	"bytes"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/require"

	goji "goji.io"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/goracle/config"
)

func init() {
	dsn := "sqlite3://file::memory:?mode=memory&cache=shared"
	// dsn = "sqlite3:///tmp/goracle.sqlite"
	catalog.SetupDB(dsn)
	catalog.AddBaseServiceTypes()
}

var apiTests = []struct {
	name    string
	method  string
	urlpath string
	in      string
	out     string
	status  int
}{
	{
		name:    "create new service",
		method:  "POST",
		urlpath: "/api/v1/services/",
		in:      `{"name": "foosvcname", "type": "internal-ops", "state": "Active"}`,
		out:     `{"components":[], "id":1, "name":"foosvcname", "description":"", "known":false, "type": "internal-ops", "environment":"", "availability_objective":"99.9", "state": "Active"}`,
		status:  http.StatusCreated,
	},
	{
		name:    "create dupe service (should fail)",
		method:  "POST",
		urlpath: "/api/v1/services/",
		in:      `{"name": "foosvcname", "type": "internal-ops"}`,
		out:     `{"errors": ["UNIQUE constraint failed: services.name"]}`,
		status:  http.StatusUnprocessableEntity,
	},
	{
		name:    "get services",
		method:  "GET",
		urlpath: "/api/v1/services/",
		in:      ``,
		out:     `[{"environment":"", "components": [], "id":1, "name":"foosvcname", "description":"", "known":false, "type": "internal-ops","availability_objective":"99.9", "state": "Active"}]`,
		status:  http.StatusOK,
	},
	{
		name:    "update service",
		method:  "PUT",
		urlpath: "/api/v1/services/1",
		in:      `{"name": "foosvcname2", "type": "internal-ops", "state": "Inactive"}`,
		out:     `{"components":[], "id":1, "name":"foosvcname2", "description":"", "known":false, "type": "internal-ops", "environment":"", "availability_objective":"99.9", "state": "Inactive"}`,
		status:  http.StatusOK,
	},
	{
		name:    "delete service",
		method:  "DELETE",
		urlpath: "/api/v1/services/1",
		in:      ``,
		out:     ``,
		status:  http.StatusOK,
	},
	{
		name:    "Create a new service for audits",
		method:  "POST",
		urlpath: "/api/v1/services",
		in:      `{"name": "service_to_audit", "type": "internal-ops", "state": "Active"}`,
		out:     `{"components":[], "id":2, "name":"service_to_audit", "description":"", "known":false, "type": "internal-ops", "environment":"", "availability_objective":"99.9", "state": "Active"}`,
		status:  http.StatusCreated,
	},
	{
		name:    "Create a owner service audit for the service",
		method:  "POST",
		urlpath: "/api/v1/services/2/audits",
		in:      `{"audit_type":"owner","audit_time":"2017-05-19T20:42:42Z","auditor":"anubhaws","action":"invalidated"}`,
		out:     `{"id":1,"service_id":2,"audit_type":"owner","audit_time":"2017-05-19T20:42:42Z","auditor":"anonymous","action":"invalidated"}`,
		status:  http.StatusCreated,
	},
	{
		name:    "Create an invalid service audit for the service",
		method:  "POST",
		urlpath: "/api/v1/services/2/audits",
		in:      `{"audit_type":"bogus","audit_time":"2017-05-19T20:42:42Z","auditor":"anubhaws"}`,
		out:     `{"errors": ["audit_type cannot be omitted or invalid"]}`,
		status:  http.StatusUnprocessableEntity,
	},
	{
		name:    "Create an audit without an auditor (should default to anon)",
		method:  "POST",
		urlpath: "/api/v1/services/2/audits",
		in:      `{"audit_type":"state","audit_time":"2017-05-19T19:42:42Z"}`,
		out:     `{"id":2,"service_id":2,"audit_type":"state","audit_time":"2017-05-19T19:42:42Z","auditor":"anonymous","action":"validated"}`,
		status:  http.StatusCreated,
	},
	{
		name:    "Create an audit with a more recent timestamp",
		method:  "POST",
		urlpath: "/api/v1/services/2/audits",
		in:      `{"audit_type":"owner","audit_time":"2017-05-19T21:42:42Z","auditor":"anubhaws","action":"expired"}`,
		out:     `{"id":3,"service_id":2,"audit_type":"owner","audit_time":"2017-05-19T21:42:42Z","auditor":"anonymous","action":"expired"}`,
		status:  http.StatusCreated,
	},
	{
		name:    "Query recent audits for a service",
		method:  "GET",
		urlpath: "/api/v1/services/2/audits",
		in:      ``,
		out:     `[{"audit_type":"owner", "auditor":"anonymous", "id":3, "service_id":2, "action":"expired", "audit_time":"2017-05-19T21:42:42Z"},{"id":2, "service_id":2, "action":"validated", "audit_time":"2017-05-19T19:42:42Z", "audit_type":"state", "auditor":"anonymous"}]`,
		status:  http.StatusOK,
	},
	{
		name:    "Query all audits for a service",
		method:  "GET",
		urlpath: "/api/v1/services/2/audits?all=true",
		in:      ``,
		out:     `[{"audit_type":"owner", "auditor":"anonymous", "id":3, "service_id":2, "action":"expired", "audit_time":"2017-05-19T21:42:42Z"},{"audit_type":"owner", "auditor":"anonymous", "id":1, "service_id":2, "action":"invalidated", "audit_time":"2017-05-19T20:42:42Z"},{"audit_type":"state", "auditor":"anonymous", "id":2, "service_id":2, "action":"validated", "audit_time":"2017-05-19T19:42:42Z"}]`,
		status:  http.StatusOK,
	},
	{
		name:    "Query all audits for a service of a particular type (should sort by timestamp, desc)",
		method:  "GET",
		urlpath: "/api/v1/services/2/audits?all=true&type=owner",
		in:      ``,
		out:     `[{"auditor":"anonymous", "id":3, "service_id":2, "action":"expired", "audit_time":"2017-05-19T21:42:42Z", "audit_type":"owner"},{"action":"invalidated", "audit_time":"2017-05-19T20:42:42Z", "audit_type":"owner", "auditor":"anonymous", "id":1, "service_id":2}]`,
		status:  http.StatusOK,
	},
	{
		name:    "Query the latest audit for a service of a particular type",
		method:  "GET",
		urlpath: "/api/v1/services/2/audits?type=owner",
		in:      ``,
		out:     `[{"auditor":"anonymous", "id":3, "service_id":2, "action":"expired", "audit_time":"2017-05-19T21:42:42Z", "audit_type":"owner"}]`,
		status:  http.StatusOK,
	},
	// here we expect the log of all changes we made so far. we
	// have to filter item_label because it has the timestamp
	// after service deletes
	{
		name:    "Query the logrecords",
		method:  "GET",
		urlpath: "/api/v1/logs/",
		in:      ``,
		out: `[
{"action":"create", "author":"", "id":4, "item_id":2, "item_type":"service"},
{"action":"delete", "author":"", "id":3, "item_id":1, "item_type":"service"},
{"action":"update", "author":"", "id":2, "item_id":1, "item_type":"service"},
{"action":"create", "author":"", "id":1, "item_id":1, "item_type":"service"}
]`,
		status: http.StatusOK,
	},
	{
		name:    "Query logrecords with filter ",
		method:  "GET",
		urlpath: "/api/v1/logs/?item_type=service&item_id=1",
		in:      ``,
		out: `[
{"action":"delete", "author":"", "id":3, "item_id":1, "item_type":"service"},
{"action":"update", "author":"", "id":2, "item_id":1, "item_type":"service"},
{"action":"create", "author":"", "id":1, "item_id":1, "item_type":"service"}
]`,
		status: http.StatusOK,
	},
	{
		name:    "Query logrecords with item_type filter, offset and limit",
		method:  "GET",
		urlpath: "/api/v1/logs/?item_type=service&offset=1&limit=2",
		in:      ``,
		out: `[
{"action":"delete", "author":"", "id":3, "item_id":1, "item_type":"service"},
{"action":"update", "author":"", "id":2, "item_id":1, "item_type":"service"}
]`,
		status: http.StatusOK,
	},
	{
		name:    "Query logrecords with item_type and action filters, offset and limit",
		method:  "GET",
		urlpath: "/api/v1/logs/?item_type=service&action=create&offset=1&limit=1",
		in:      ``,
		out: `[
{"action":"create", "author":"", "id":1, "item_id":1, "item_type":"service"}
]`,
		status: http.StatusOK,
	},
	{
		name:    "Create Attribute",
		method:  "POST",
		urlpath: "/api/v2/query",
		in:      `{"query":"mutation($attribute:AttributeInput!){createAttribute(attribute:$attribute) {\n  id\n  name\n  value\n  object_id\n  object_type\n}}","variables":{"attribute":{"name":"test","value":"","object_type":"component","object_id":"2"}}}`,
		out:     `{"error":"Mutation query is forbidden"}`,
		status:  http.StatusForbidden,
	},
	{
		name:    "Get Attributes",
		method:  "POST",
		urlpath: "/api/v2/query",
		in:      `{"query":"{\n  attributes {\n    id\n    name\n    value\n    object_type\n    object_id\n  }\n}\n","variables":null,"operationName":null}`,
		out:     `{"data":{"attributes":[]}}`,
		status:  http.StatusOK,
	},
	{
		name:    "Update Attribute",
		method:  "POST",
		urlpath: "/api/v2/query",
		in:      `{"query":"mutation ($attribute: AttributeInput!) {\n  updateAttribute(id: \"1\", attribute: $attribute) {\n    id\n    name\n    value\n    object_id\n    object_type\n  }\n}\n","variables":{"attribute":{"name":"test","value":"test","object_type":"component","object_id":"2"}}}`,
		out:     `{"error":"Mutation query is forbidden"}`,
		status:  http.StatusForbidden,
	},
	{
		name:    "Delete Attribute",
		method:  "POST",
		urlpath: "/api/v2/query",
		in:      `{"query":"mutation{deleteAttribute(id:\"1\") {\n  id\n  name\n  value\n  object_id\n  object_type\n}}","variables":null}`,
		out:     `{"error":"Mutation query is forbidden"}`,
		status:  http.StatusForbidden,
	},
}

// jsonStrip strips a list of tags from a json blob
// it only handles objects and arrays of objects. it only strips
// values from the top level object or the top level objects in
// the array
func jsonStrip(t *testing.T, b []byte, tags ...string) []byte {
	if len(b) < 1 {
		t.Fatalf("Got 0 length input in jsonStrip: %s", string(b))
	}
	switch b[0] {
	case '{':
		j := map[string]interface{}{}
		err := json.Unmarshal(b, &j)
		require.NoError(t, err, "Coudn't unmarshal JSON response for stripping: %s", string(b))
		for _, tag := range tags {
			delete(j, tag)
		}
		bout, err := json.Marshal(j)
		require.NoError(t, err, "Coudn't re-marshal JSON response after stripping: %s", string(b))
		return bout
	case '[':
		j := []map[string]interface{}{}
		err := json.Unmarshal(b, &j)
		require.NoError(t, err, "Coudn't unmarshal JSON response for stripping: %s", string(b))
		for _, v := range j {
			for _, tag := range tags {
				delete(v, tag)
			}
		}
		bout, err := json.Marshal(j)
		require.NoError(t, err, "Coudn't re-marshal JSON response after stripping: %s", string(b))
		return bout

	}
	t.Fatalf("jsonStrip got string that doesn't start with [ or {: %s", string(b))
	return nil
}

func TestAPIIntegration(t *testing.T) {
	config.Config.EnableS2S = false
	router := &statusServer{goji.NewMux()}
	router.addHandlers("./assets", false)

	for _, tt := range apiTests {
		if testing.Verbose() {
			t.Logf("Starting test case %s", tt.name)
		}
		req, err := http.NewRequest(tt.method, tt.urlpath, bytes.NewBufferString(tt.in))
		require.NoError(t, err, "NewRequest fail for: %s", tt.name)
		w := httptest.NewRecorder()
		router.ServeHTTP(w, req)
		require.Equal(t, tt.status, w.Code, "Wrong HTTP status for: %s", tt.name)
		bod := w.Body.Bytes()
		if len(tt.out) > 0 && (tt.out[0] == '{' || tt.out[0] == '[') {
			// we strip created and updated because they
			// are time dependent
			bod = jsonStrip(t, bod, "created", "updated", "service", "component", "before", "after", "item_label")
			require.JSONEq(t, tt.out, string(bod), "Returned JSON doesn't match for: %s", tt.name)
		} else {
			require.Equal(t, tt.out, string(bod), "Returned string doesn't match for: %s", tt.name)
		}

	}
}
