package cataloghandlers

import (
	"encoding/json"
	"fmt"
	"time"

	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
	"strings"
)

// Component is the API visible version of catalog components
type Component struct {
	// Description  string
	ID          *uint    `json:"id"`
	Label       *string  `json:"label"`
	Claimed     *bool    `json:"claimed"`
	Name        *string  `json:"name"`
	Description *string  `json:"description"`
	Type        *string  `json:"type"`
	Metrics     *[]uint  `json:"metric_ids"`
	Rollup      *bool    `json:"rollup"`
	ServiceID   *uint    `json:"service_id"`
	Service     *Service `json:"service,omitempty"`
}

// DumpToDBComponent takes a DB Component struct and dumps the API
// Component struct fields into it after proper conversions and
// returns the modified DB Component Performs overwrites
func (ac *Component) DumpToDBComponent(dc *catalog.Component) (*catalog.Component, error) {
	//var err error

	// Go through each field and set updates
	if ac.Label != nil {
		dc.Label = *ac.Label
	}
	if ac.Claimed != nil {
		dc.Claimed = *ac.Claimed
	}
	if ac.Name != nil {
		dc.Name = *ac.Name
	}
	if ac.Description != nil {
		dc.Description = *ac.Description
	}
	if ac.Type != nil {
		dc.Type = *ac.Type
	}
	if ac.Metrics != nil {
		// First clear the Metric currently on the Component
		dc.Metrics = make([]*catalog.Metric, 0)
		for _, mid := range *ac.Metrics {
			met, err := catalog.GetCatalog().GetMetricByID(mid)
			if err != nil {
				return nil, err
			}
			dc.Metrics = append(dc.Metrics, met)
		}
	}
	if ac.Rollup != nil {
		dc.Rollup = *ac.Rollup
	}
	if ac.ServiceID != nil {
		dc.ServiceID = *ac.ServiceID
	}
	return dc, nil
}

// LoadFromDBComponent will load attributes from a Component
// in the DB table into this API Component struct, making
// necessary conversions and modifying the given
// component `c` in-place
func (ac *Component) LoadFromDBComponent(dc *catalog.Component) error {
	//var err error
	ac.ID = &dc.ID
	ac.Type = &dc.Type
	ac.Label = &dc.Label
	ac.Claimed = &dc.Claimed
	ac.Name = &dc.Name
	ac.Description = &dc.Description
	ac.Rollup = &dc.Rollup
	// API Component structs only expose the metric ID, not the deep nested Metric data
	metIds := make([]uint, 0)
	for _, met := range dc.Metrics {
		metIds = append(metIds, met.ID)
	}
	ac.Metrics = &metIds
	ac.ServiceID = &dc.ServiceID
	if ac.ServiceID != nil && dc.Service != nil {
		s := &Service{}
		s.LoadFromDBService(dc.Service)
		ac.Service = s
	}
	return nil
}

// Service is the API visible version of catalog services
type Service struct {
	ID                    *uint      `json:"id"`
	Name                  *string    `json:"name"` // Globally unique human service name
	Description           *string    `json:"description"`
	Known                 *bool      `json:"known"`
	Type                  *string    `json:"type"`
	Environment           *string    `json:"environment"`
	Components            *[]uint    `json:"components"`
	AvailabilityObjective *float64   `json:"availability_objective,string"`
	CreatedAt             *time.Time `json:"created"`
	UpdatedAt             *time.Time `json:"updated"`
	State                 *string    `json:"state"`
}

// LoadFromDBService loads a Service from the Catalog DB and
// transfers the fields into an API Service struct
func (as *Service) LoadFromDBService(ds *catalog.Service) error {
	as.ID = &ds.ID
	as.Name = &ds.Name
	as.Description = &ds.Description
	as.Known = &ds.Known
	if ds.ServiceType != nil {
		as.Type = &ds.ServiceType.Label
	}
	as.Environment = &ds.Environment
	compIds := make([]uint, 0)
	for _, comp := range ds.Components {
		compIds = append(compIds, comp.ID)
	}
	as.Components = &compIds
	as.CreatedAt = &ds.CreatedAt
	as.UpdatedAt = &ds.UpdatedAt
	as.AvailabilityObjective = &ds.AvailabilityObjective
	if ds.State != nil {
		as.State = &ds.State.Name
	}
	return nil
}

// DumpToDBService takes a DB Service struct and dumps the API
// Service struct fields into it after proper conversions and
// returns the modified DB Service Performs overwrites
func (as *Service) DumpToDBService(ds *catalog.Service) error {
	// Go through each field and set updates
	if as.Known != nil {
		ds.Known = *as.Known
	}
	if as.Description != nil {
		ds.Description = *as.Description
	}
	if as.Name != nil {
		ds.Name = *as.Name
	}
	if as.Environment != nil {
		ds.Environment = *as.Environment
	}
	// Do a translation from a string into a ServiceType
	if as.Type != nil {
		serviceTypes, err := catalog.GetServiceTypeMap()
		if err != nil {
			return err
		}
		serviceType, found := serviceTypes[*as.Type]
		if !found {
			return fmt.Errorf("no such service type '%s'", *as.Type)
		}
		// This should be temporary until we can get rid of the
		// ServiceType table
		ds.ServiceTypeID = serviceType.ID
		ds.ServiceType = serviceType
	}
	if as.Components != nil {
		// First clear the Components currently on the Service
		ds.Components = make([]*catalog.Component, 0)
		for _, cid := range *as.Components {
			comp, err := catalog.GetCatalog().GetComponentByID(cid)
			if err != nil {
				return err
			}
			ds.Components = append(ds.Components, comp)
		}
	}
	if as.AvailabilityObjective != nil {
		ds.AvailabilityObjective = *as.AvailabilityObjective
	}
	if as.State != nil {
		states, err := catalog.GetServiceStateMap()
		if err != nil {
			return err
		}
		state, found := states[strings.ToLower(*as.State)]
		if !found {
			return fmt.Errorf("no such service state '%s'", *as.State)
		}
		ds.State = state
		ds.StateID = state.ID
	}
	// we don't populate CreatedAt and UpdatedAt since these are
	// managed automatically by gorm
	return nil
}

// Metric is the API visible version of catalog metrics
type Metric struct {
	ID              *uint      `json:"id"`
	Label           *string    `json:"label"`
	Name            *string    `json:"name"`
	Description     *string    `json:"description"`
	AutoGenerated   *bool      `json:"autogenerated"`
	Queries         []uint     `json:"queries"`
	Threshold       *float64   `json:"threshold"`
	CalculationType *string    `json:"calculation_type"`
	ComponentRollup *bool      `json:"component_rollup"`
	ComponentID     *uint      `json:"component_id"`
	Component       *Component `json:"component,omitempty"`
	// FeatureRollup   *bool    `json:"feature_rollup"`
	LatencyQuery     *string  `json:"latency_query"`
	LatencyObjective *float64 `json:"latency_objective"`
}

// DumpToDBMetric takes a DB Metric struct and dumps the API
// Metric struct fields into it after proper conversions and
// returns the modified DB Metric Performs overwrites
func (am *Metric) DumpToDBMetric(dm *catalog.Metric) (*catalog.Metric, error) {
	if am.ID != nil {
		dm.ID = *am.ID
	}
	if am.Label != nil {
		dm.Label = *am.Label
	}
	if am.Name != nil {
		dm.Name = *am.Name
	}
	if am.Description != nil {
		dm.Description = *am.Description
	}
	if am.ComponentRollup != nil {
		dm.ComponentRollup = *am.ComponentRollup
	}
	if am.AutoGenerated != nil {
		dm.AutoGenerated = *am.AutoGenerated
	}
	if am.Threshold != nil {
		dm.Threshold = *am.Threshold
	}
	if am.CalculationType != nil {
		dm.CalculationType = catalog.CalculationType(*am.CalculationType)
	}
	if am.LatencyQuery != nil {
		if dm.CalculationType == catalog.Latency {
			dm.LatencyQuery = catalog.QueryType(*am.LatencyQuery)
		}
	}

	if am.Queries != nil {
		dm.Queries = make([]*catalog.Query, len(am.Queries))
		cat := catalog.GetCatalog()
		for i, qid := range am.Queries {
			q, err := cat.GetQueryByID(qid)
			if err != nil {
				return nil, err
			}
			dm.Queries[i] = q

		}
	}

	if am.ComponentID != nil {
		dm.ComponentID = *am.ComponentID
	}

	// TODO: handle features
	return dm, nil
}

// LoadFromDBMetric loads a Metric from the Catalog DB and
// transfers the fields into an API Metric struct
func (am *Metric) LoadFromDBMetric(dm *catalog.Metric) error {
	am.ID = &dm.ID
	am.Label = &dm.Label
	am.Name = &dm.Name
	am.Description = &dm.Description
	am.ComponentRollup = &dm.ComponentRollup
	am.AutoGenerated = &dm.AutoGenerated
	am.Threshold = &dm.Threshold
	c := string(dm.CalculationType)
	am.CalculationType = &c
	l := string(dm.LatencyQuery)
	am.LatencyQuery = &l
	am.LatencyObjective = &dm.LatencyObjective
	am.Queries = make([]uint, len(dm.Queries))
	for i, q := range dm.Queries {
		am.Queries[i] = q.ID
	}
	am.ComponentID = &dm.ComponentID
	// Load up a nested component
	if am.ComponentID != nil && dm.Component != nil {
		c := &Component{}
		c.LoadFromDBComponent(dm.Component)
		am.Component = c
	}
	// TODO: handle features
	return nil
}

// Query is the API visible version of catalog queries
type Query struct {
	ID            *uint   `json:"id"`
	Type          *string `json:"type"`
	Query         *string `json:"query"`
	AggregateType *string `json:"aggregate_type"`
}

// DumpToDBQuery takes a DB Query struct and dumps the API
// Query struct fields into it after proper conversions and
// returns the modified DB Query Performs overwrites
func (aq *Query) DumpToDBQuery(dq *catalog.Query) (*catalog.Query, error) {
	if aq.ID != nil {
		dq.ID = *aq.ID
	}
	if aq.Type != nil {
		dq.QueryType = catalog.QueryType(*aq.Type)
	}
	if aq.Query != nil {
		dq.Query = *aq.Query
	}
	if aq.AggregateType != nil {
		dq.AggregateType = catalog.AggregateType(*aq.AggregateType)
	}
	return dq, nil
}

// LoadFromDBQuery loads a Query from the Catalog DB and
// transfers the fields into an API Query struct
func (aq *Query) LoadFromDBQuery(dq *catalog.Query) error {
	aq.ID = &dq.ID
	qtype := string(dq.QueryType)
	aq.Type = &qtype
	aq.Query = &dq.Query
	atype := string(dq.AggregateType)
	aq.AggregateType = &atype
	return nil
}

// ServiceAudit is the API visible version of catalog service audit actions
type ServiceAudit struct {
	ID        *uint      `json:"id"`
	ServiceID *uint      `json:"service_id"`
	AuditType *string    `json:"audit_type"`
	Auditor   *string    `json:"auditor"`
	Action    *string    `json:"action"`
	AuditTime *time.Time `json:"audit_time"`
}

// DumpToDBServiceAudit takes a DB ServiceAudit struct and dumps the API
// struct fields into it after proper conversions and
// returns the modified DB ServiceAudit Performs overwrites
func (asa *ServiceAudit) DumpToDBServiceAudit(dsa *catalog.ServiceAudit) (*catalog.ServiceAudit, error) {
	if asa.ID != nil {
		dsa.ID = *asa.ID
	}
	if asa.ServiceID != nil {
		dsa.ServiceID = *asa.ServiceID
	}
	if asa.AuditType != nil {
		dsa.AuditType = *asa.AuditType
	}
	if asa.Auditor != nil {
		dsa.Auditor = *asa.Auditor
	}
	if asa.AuditTime != nil {
		dsa.AuditTime = *asa.AuditTime
	}
	if asa.Action != nil {
		dsa.Action = *asa.Action
	}

	return dsa, nil
}

// LoadFromDBServiceAudit loads a ServiceAudit from the Catalog DB and
// transfers the fields into an API ServiceAudit struct
func (asa *ServiceAudit) LoadFromDBServiceAudit(dsa *catalog.ServiceAudit) error {
	asa.ID = &dsa.ID
	asa.ServiceID = &dsa.ServiceID
	asa.AuditType = &dsa.AuditType
	asa.Auditor = &dsa.Auditor
	asa.AuditTime = &dsa.AuditTime
	asa.Action = &dsa.Action

	return nil
}

type Feature struct {
	ID       *uint   `json:"id"`
	Label    *string `json:"sting"`
	Name     *string `json:"name"`
	Metrics  *[]uint `json:"metrics"`
	Children *[]uint `json:"children"`
	Parent   *uint   `json:"parent"`
}

// LoadFromDBServiceAudit loads a ServiceAudit from the Catalog DB and
// transfers the fields into an API ServiceAudit struct
func (af *Feature) LoadFromDBFeature(df *catalog.Feature) error {
	af.ID = &df.ID
	af.Label = &df.Label
	af.Name = &df.Name
	af.Parent = df.FeatureID

	metIds := make([]uint, 0)
	for _, met := range df.Metrics {
		metIds = append(metIds, met.ID)
	}
	af.Metrics = &metIds
	childIds := make([]uint, 0)
	for _, met := range df.Children {
		childIds = append(childIds, met.ID)
	}
	af.Children = &childIds

	return nil
}

type LogRecord struct {
	ID        uint        `json:"id"`
	Created   time.Time   `json:"created"`
	Action    string      `json:"action"`
	ItemType  string      `json:"item_type"`
	ItemLabel string      `json:"item_label"`
	ItemID    uint        `json:"item_id"`
	Before    interface{} `json:"before"`
	After     interface{} `json:"after"`
	Author    string      `json:"author"`
}

func (alr *LogRecord) LoadFromDBLogRecord(dlr *catalog.LogRecord) error {
	alr.ID = dlr.ID
	alr.Created = dlr.CreatedAt
	alr.Action = dlr.Action
	alr.ItemType = dlr.ItemType
	alr.ItemLabel = dlr.ItemLabel
	alr.ItemID = dlr.ItemID
	if dlr.Before != "" {
		err := json.Unmarshal([]byte(dlr.Before), &alr.Before)
		if err != nil {
			// return fmt.Errorf("Failed to unmarshal Before for logrecord %d: %s", dlr.ID, err)
			logrus.Errorf("Failed to unmarshal Before for logrecord %d, returning it as string: %s", dlr.ID, err)
			alr.Before = dlr.Before
		}
	}
	if dlr.After != "" {
		err := json.Unmarshal([]byte(dlr.After), &alr.After)
		if err != nil {
			// return fmt.Errorf("Failed to unmarshal After for logrecord %d: %s", dlr.ID, err)
			logrus.Errorf("Failed to unmarshal After for logrecord %d, returning it as string: %s", dlr.ID, err)
			alr.After = dlr.After
		}
	}
	alr.Author = dlr.Author
	return nil
}
