package cataloghandlers

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"strconv"
	"time"

	"goji.io"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/goracle/guardianauth"
	"code.justin.tv/availability/goracle/serverutil"
	"github.com/rs/cors"
	"github.com/sirupsen/logrus"
	"goji.io/pat"
	"code.justin.tv/availability/goracle/goracleUser"
)

// TODO: REMOVE TEMPORARY GRAPHQL HELPER

// RegisterHandlers registers catalog API handlers
func RegisterHandlers(mux *goji.Mux) {
	logrus.Printf("WARNING: registering handlers for legacy apis at /api/v1")
	api := goji.SubMux()

	c := cors.New(cors.Options{
		AllowedOrigins: []string{"http://localhost:3000", "http://servicecatalog-local.internal.justin.tv:3000", "https://servicecatalog-dev.internal.justin.tv", "https://servicecatalog.internal.justin.tv"},
		AllowedMethods: []string{"GET", "POST", "HEAD", "OPTIONS", "PUT", "DELETE", "PATCH"},
		AllowedHeaders:   []string{"*"},
		MaxAge: 86400,
		AllowCredentials: true,
	})

	api.Use(c.Handler)
	api.Use(guardianauth.GuardianUser)
	mux.Handle(pat.New("/api/v1/*"), api)

	// COMPONENTS
	api.HandleFunc(pat.Get("/components/"), getComponents)
	api.HandleFunc(pat.Get("/components"), getComponents)
	api.HandleFunc(pat.Get("/components/:id"), getComponent)
	api.HandleFunc(pat.Post("/components/"), postComponent)
	api.HandleFunc(pat.Post("/components"), postComponent)
	api.HandleFunc(pat.Put("/components/:id"), putComponent)
	api.HandleFunc(pat.Delete("/components/:id"), deleteComponent)

	api.HandleFunc(pat.Post("/services/:id/audits"), postServiceAudit)
	api.HandleFunc(pat.Get("/services/:id/audits"), getServiceAudits)
	api.HandleFunc(pat.Post("/services/"), postService)
	api.HandleFunc(pat.Post("/services"), postService)
	api.HandleFunc(pat.Get("/services/"), getServices)
	api.HandleFunc(pat.Get("/services"), getServices)
	api.HandleFunc(pat.Get("/services/:id"), getService)
	api.HandleFunc(pat.Put("/services/:id"), putService)
	api.HandleFunc(pat.Delete("/services/:id"), deleteService)

	api.HandleFunc(pat.Get("/metrics/"), getMetrics)
	api.HandleFunc(pat.Get("/metrics/:id"), getMetric)
	api.HandleFunc(pat.Post("/metrics/"), postMetric)
	api.HandleFunc(pat.Post("/metrics"), postMetric)
	api.HandleFunc(pat.Put("/metrics/:id"), putMetric)
	api.HandleFunc(pat.Delete("/metrics/:id"), deleteMetric)

	api.HandleFunc(pat.Get("/queries/"), getQueries)
	api.HandleFunc(pat.Post("/queries/"), postQuery)
	api.HandleFunc(pat.Post("/queries"), postQuery)
	api.HandleFunc(pat.Get("/queries/:id"), getQuery)
	api.HandleFunc(pat.Put("/queries/:id"), putQuery)
	api.HandleFunc(pat.Delete("/queries/:id"), deleteQuery)

	api.HandleFunc(pat.Get("/servicetypes/"), getServiceTypes)
	api.HandleFunc(pat.Get("/unknowncomponents/"), getUnknownComponents)

	api.HandleFunc(pat.Get("/features/"), getFeatures)
	api.HandleFunc(pat.Get("/features"), getFeatures)
	api.HandleFunc(pat.Get("/features/:id"), getFeature)

	api.HandleFunc(pat.Get("/logs/"), getLogs)
}

func getServiceTypes(w http.ResponseWriter, r *http.Request) {
	servTypes, err := catalog.GetCatalog().GetServiceTypes()
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
	}
	constructResponse(w, http.StatusOK, servTypes)
}

func getUnknownComponents(w http.ResponseWriter, r *http.Request) {
	comps := catalog.GetCatalog().UnknownComponents()
	constructResponse(w, http.StatusOK, comps)
}

func constructResponse(w http.ResponseWriter, status int, out interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(status)
	if err := json.NewEncoder(w).Encode(out); err != nil {
		panic(err)
	}
}

// CONVERTED API HANDLERS

// COMPONENTS
func postComponent(w http.ResponseWriter, r *http.Request) {
	var apiComp Component
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{errors.New("internal server error reading HTTP POST body")}, 500)
		return
	}
	defer r.Body.Close()
	// unmarshal into an API component object
	if err := json.Unmarshal(body, &apiComp); err != nil {
		logrus.Warnf("Unable to unmarshal POST data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	if apiComp.ID != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("cannot set 'id' when creating a component")}, 400)
		return
	}

	var dbComp *catalog.Component
	dbComp = catalog.DefaultComponent()
	dbComp, err = apiComp.DumpToDBComponent(dbComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Create the Component and save to DB
	err = catalog.GetCatalog().AddComponent(dbComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}

	// Any additional fields set through adding the Component
	// (e.g. the ID) should be included in the API struct handed back
	// so we need to load them into the original apiComp
	apiComp.LoadFromDBComponent(dbComp)

	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, dbComp, r.Context())

	returnJSON, err := json.Marshal(apiComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusCreated)
	w.Write(returnJSON)

}

func getComponents(w http.ResponseWriter, r *http.Request) {
	lookupMap := make(map[string]interface{})
	// lookup supported URL query params
	label := r.URL.Query().Get("label")
	if label != "" {
		lookupMap["label"] = label
	}
	// support lookups on `service_id`
	sid := r.URL.Query().Get("service_id")
	if sid != "" {
		lookupMap["service_id"] = sid
	}

	dbComps, err := catalog.GetCatalog().GetComponents(lookupMap)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	apiComps := make([]*Component, 0)
	// Marshall all of the DB components into API responses
	for _, dbComp := range dbComps {
		apiComp := &Component{}
		err := apiComp.LoadFromDBComponent(dbComp)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiComps = append(apiComps, apiComp)
	}
	returnJSON, err := json.Marshal(apiComps)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func getComponent(w http.ResponseWriter, r *http.Request) {
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid component id: %s", err.Error())}, 400)
		return
	}
	dbComp, err := catalog.GetCatalog().GetComponentByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}
	apiComp := &Component{}
	err = apiComp.LoadFromDBComponent(dbComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	returnJSON, err := json.Marshal(apiComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func putComponent(w http.ResponseWriter, r *http.Request) {
	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid component id: %s", err.Error())}, 400)
		return
	}
	// Get the component
	dbComp, err := catalog.GetCatalog().GetComponentByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	// Parse the fields from the API request
	// TODO: change ioutil.ReadAll to a json Decoder
	var apiComp Component
	bl := dbComp.LogInfo()
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{errors.New("internal server error reading HTTP POST body")}, 500)
		return
	}
	defer r.Body.Close()
	// Unmarshal API request fields into API struct
	if err := json.Unmarshal(body, &apiComp); err != nil {
		logrus.Warnf("Unable to unmarshal POST data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// If the API request includes the ID of the component,
	// it better be the same as the id in the URL
	if apiComp.ID != nil && *apiComp.ID != id {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("json component id, if specified, must match URL id")}, 400)
		return
	}

	// Dump the API request fields into the Component
	dbComp, err = apiComp.DumpToDBComponent(dbComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Create the Component and save to DB
	err = catalog.GetCatalog().AddComponent(dbComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}

	// Marshal the most up to date info about dbComp into
	// the API struct that will be returned as a response
	apiComp.LoadFromDBComponent(dbComp)
	al := dbComp.LogInfo()
	returnJSON, err := json.Marshal(apiComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, bl, al, r.Context())
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func deleteComponent(w http.ResponseWriter, r *http.Request) {
	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid component id: %s", err.Error())}, 400)
		return
	}

	// Get the component
	dbComp, err := catalog.GetCatalog().GetComponentByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	err = catalog.GetCatalog().DeleteComponent(dbComp)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, dbComp, nil, r.Context())

	w.WriteHeader(http.StatusOK)
}

// SERVICES

func getService(w http.ResponseWriter, r *http.Request) {
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		return
	}
	dbServ, err := catalog.GetCatalog().GetServiceByID(uint(id))
	// Marshal the DB record into an API struct
	apiServ := &Service{}
	err = apiServ.LoadFromDBService(dbServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	returnJSON, err := json.Marshal(apiServ)
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func getServices(w http.ResponseWriter, r *http.Request) {
	dbServs, err := catalog.GetCatalog().GetServices(nil)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	// Convert the services to an API format
	apiServs := make([]*Service, 0)
	// Marshall all of the DB components into API responses
	for _, dbServ := range dbServs {
		apiServ := &Service{}
		err := apiServ.LoadFromDBService(dbServ)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiServs = append(apiServs, apiServ)
	}

	returnJSON, err := json.Marshal(apiServs)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func postService(w http.ResponseWriter, r *http.Request) {
	var apiServ Service
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{errors.New("internal server error reading HTTP POST body")}, 500)
		return
	}
	defer r.Body.Close()
	// unmarshal into an API service object
	if err := json.Unmarshal(body, &apiServ); err != nil {
		logrus.Warnf("Unable to unmarshal POST data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	if apiServ.ID != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("cannot set 'id' when creating a component")}, 400)
		return
	}

	var dbServ *catalog.Service
	dbServ = catalog.DefaultService()
	err = apiServ.DumpToDBService(dbServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Create the Service and save to DB
	err = catalog.GetCatalog().AddService(dbServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}

	// Any additional fields set through adding the Servuce
	// (e.g. the ID) should be included in the API struct handed back
	// so we need to load them into the original apiServ
	apiServ.LoadFromDBService(dbServ)

	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, dbServ, r.Context())

	returnJSON, err := json.Marshal(apiServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusCreated)
	w.Write(returnJSON)
}

func putService(w http.ResponseWriter, r *http.Request) {
	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid service id: %s", err.Error())}, 400)
		return
	}
	// Get the service
	dbServ, err := catalog.GetCatalog().GetServiceByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	var apiServ Service
	bl := dbServ.LogInfo()

	// Parse the fields from the API request
	// TODO: change ioutil.ReadAll to a json Decoder
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{errors.New("internal server error reading HTTP POST body")}, 500)
		return
	}
	defer r.Body.Close()
	// Unmarshal API request fields into API struct
	if err := json.Unmarshal(body, &apiServ); err != nil {
		logrus.Warnf("Unable to unmarshal POST data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// If the API request includes the ID of the service,
	// it better be the same as the id in the URL
	if apiServ.ID != nil && *apiServ.ID != id {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("json service id, if specified, must match URL id")}, 400)
		return
	}

	// Dump the API request fields into the Service
	err = apiServ.DumpToDBService(dbServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Create the Component and save to DB
	err = catalog.GetCatalog().AddService(dbServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}

	// Marshal the most up to date info about dbComp into
	// the API struct that will be returned as a response
	apiServ.LoadFromDBService(dbServ)

	al := dbServ.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, bl, al, r.Context())

	returnJSON, err := json.Marshal(apiServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func deleteService(w http.ResponseWriter, r *http.Request) {
	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid service id: %s", err.Error())}, 400)
		return
	}

	// Get the service
	dbServ, err := catalog.GetCatalog().GetServiceByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	err = catalog.GetCatalog().DeleteService(dbServ)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, dbServ, nil, r.Context())

	w.WriteHeader(http.StatusOK)
}

// METRICS

func getMetrics(w http.ResponseWriter, r *http.Request) {
	lookupMap := make(map[string]interface{})
	ct := r.URL.Query().Get("calculation_type")
	if ct != "" {
		lookupMap["calculation_type"] = ct
	}
	ct = r.URL.Query().Get("component_id")
	if ct != "" {
		lookupMap["component_id"] = ct
	}
	dbMetrics := catalog.GetCatalog().GetMetrics(lookupMap)
	apiMetrics := make([]*Metric, 0)
	// Create a map for query lookups by metric_id (see comment below)
	queriesParams := make(map[string]interface{})
	// Marshall all of the DB components into API responses
	for _, dbMetric := range dbMetrics {
		// 11/16/2017 (nherson): preloads have been disabled
		// for Metric.Queries. Here, we manually load each
		// metrics queries into the db struct before continuing
		if dbMetric.Queries == nil {
			// There is an index on this field in the DB so this _should_ be a fast operation
			queriesParams["metric_id"] = dbMetric.ID
			queries, err := catalog.GetCatalog().GetQueriesComplete(nil, queriesParams)
			if err != nil {
				serverutil.HandleAPIErrors(w, []error{err}, 500)
				return
			}
			dbMetric.Queries = queries
		}

		apiMetric := &Metric{}
		err := apiMetric.LoadFromDBMetric(dbMetric)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiMetrics = append(apiMetrics, apiMetric)
	}
	returnJSON, err := json.Marshal(apiMetrics)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func getMetric(w http.ResponseWriter, r *http.Request) {
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
	}
	dbMetric, err := catalog.GetCatalog().GetMetricByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}
	apiMetric := &Metric{}
	apiMetric.LoadFromDBMetric(dbMetric)
	constructResponse(w, http.StatusOK, apiMetric)
}

func postMetric(w http.ResponseWriter, r *http.Request) {
	var apiMetric Metric

	defer r.Body.Close()

	err := json.NewDecoder(r.Body).Decode(&apiMetric)
	if err != nil {
		logrus.Warnf("Unable to unmarshal POST metric data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	if apiMetric.ID != nil {
		err := fmt.Errorf("cannot set ID when creating a metric")
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	dbMetric, err := apiMetric.DumpToDBMetric(&catalog.Metric{})
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	err = catalog.GetCatalog().AddMetric(dbMetric)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}
	apiMetric.LoadFromDBMetric(dbMetric)

	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, dbMetric, r.Context())

	constructResponse(w, http.StatusCreated, apiMetric)
}

func putMetric(w http.ResponseWriter, r *http.Request) {
	var apiMetric Metric
	defer r.Body.Close()

	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid metric id: %s", err.Error())}, 400)
		return
	}

	dbMetric, err := catalog.GetCatalog().GetMetricByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}
	bl := dbMetric.LogInfo()

	err = json.NewDecoder(r.Body).Decode(&apiMetric)
	if err != nil {
		logrus.Warnf("Unable to unmarshal POST metric data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	if apiMetric.ID != nil && *apiMetric.ID != id {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("json metric id, if specified, must match URL id")}, 400)
		return
	}

	dbMetric, err = apiMetric.DumpToDBMetric(dbMetric)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Create the Component and save to DB
	err = catalog.GetCatalog().AddMetric(dbMetric)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}
	apiMetric.LoadFromDBMetric(dbMetric)

	al := dbMetric.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, bl, al, r.Context())

	constructResponse(w, http.StatusOK, apiMetric)
}

func deleteMetric(w http.ResponseWriter, r *http.Request) {
	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid metric id: %s", err.Error())}, 400)
		return
	}

	// Get the component
	dbMetric, err := catalog.GetCatalog().GetMetricByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	err = catalog.GetCatalog().DeleteMetric(dbMetric)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, dbMetric, nil, r.Context())

	//w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
}

func getQueries(w http.ResponseWriter, r *http.Request) {
	dbQueries := catalog.GetCatalog().GetQueries(nil)
	apiQueries := make([]*Query, 0)
	for _, dbQuery := range dbQueries {
		apiQuery := &Query{}
		err := apiQuery.LoadFromDBQuery(dbQuery)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiQueries = append(apiQueries, apiQuery)
	}
	returnJSON, err := json.Marshal(apiQueries)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}
func getQuery(w http.ResponseWriter, r *http.Request) {
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
	}
	dbQuery, err := catalog.GetCatalog().GetQueryByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}
	apiQuery := &Query{}
	apiQuery.LoadFromDBQuery(dbQuery)
	constructResponse(w, http.StatusOK, apiQuery)
}

func postQuery(w http.ResponseWriter, r *http.Request) {
	var apiQuery Query

	defer r.Body.Close()

	err := json.NewDecoder(r.Body).Decode(&apiQuery)
	if err != nil {
		logrus.Warnf("Unable to unmarshal POST query data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	if apiQuery.ID != nil {
		err := fmt.Errorf("cannot set ID when creating a query")
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	dbQuery, err := apiQuery.DumpToDBQuery(&catalog.Query{})
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	err = catalog.GetCatalog().AddQuery(dbQuery)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}
	apiQuery.LoadFromDBQuery(dbQuery)

	catalog.SaveAPILoggables(catalog.LogOpCreate, nil, dbQuery, r.Context())

	constructResponse(w, http.StatusCreated, apiQuery)
}

func putQuery(w http.ResponseWriter, r *http.Request) {
	var apiQuery Query
	defer r.Body.Close()

	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid query id: %s", err.Error())}, 400)
		return
	}

	dbQuery, err := catalog.GetCatalog().GetQueryByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	bl := dbQuery.LogInfo()

	err = json.NewDecoder(r.Body).Decode(&apiQuery)
	if err != nil {
		logrus.Warnf("Unable to unmarshal POST query data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	if apiQuery.ID != nil && *apiQuery.ID != id {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("json query id, if specified, must match URL id")}, 400)
		return
	}

	dbQuery, err = apiQuery.DumpToDBQuery(dbQuery)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Create the Component and save to DB
	err = catalog.GetCatalog().AddQuery(dbQuery)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}
	apiQuery.LoadFromDBQuery(dbQuery)

	al := dbQuery.LogInfo()
	catalog.SaveAPILogInfos(catalog.LogOpUpdate, bl, al, r.Context())

	constructResponse(w, http.StatusOK, apiQuery)
}

func deleteQuery(w http.ResponseWriter, r *http.Request) {
	// Pull out the id from the request route
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid query id: %s", err.Error())}, 400)
		return
	}

	// Get the component
	dbQuery, err := catalog.GetCatalog().GetQueryByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	err = catalog.GetCatalog().DeleteQuery(dbQuery)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	catalog.SaveAPILoggables(catalog.LogOpDelete, dbQuery, nil, r.Context())

	w.WriteHeader(http.StatusOK)

}

// Service Audits

// Since this GET request is from /services/:id/audits,
// the lookup is a bit special and requires doing some
// validation that the given service exists before doing a lookup
// TODO: make a special route called /services/:id/audits/all
// that goes to this same handler but returns every single audit for
// the service instead of the most recent ones
func getServiceAudits(w http.ResponseWriter, r *http.Request) {
	lookupMap := make(map[string]interface{})

	// Get the parent service ID from the URL
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// Check for `type` and `all` labels
	auditType := r.URL.Query().Get("type")
	if auditType != "" {
		lookupMap["audit_type"] = auditType
	}
	fetchAll := false
	all := r.URL.Query().Get("all")
	if all != "" && all == "true" {
		fetchAll = true
	}

	// Make sure that the service at the specified ID actually exists
	_, err = catalog.GetCatalog().GetServiceByID(uint(id))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	// lookup supported URL query params
	lookupMap["service_id"] = id

	var dbServiceAudits []*catalog.ServiceAudit
	// Either fetch every audit for the service, or just the lastest ones
	if fetchAll {
		dbServiceAudits = catalog.GetCatalog().GetServiceAudits(lookupMap)
	} else {
		// Special DB lookup for just the most recent service audits for each type of audit
		dbServiceAudits = catalog.GetCatalog().GetLatestServiceAudits(lookupMap)
	}

	apiServiceAudits := make([]*ServiceAudit, 0)
	// Marshall all of the DB ServiceAudits into API responses
	for _, dbServiceAudit := range dbServiceAudits {
		apiServiceAudit := &ServiceAudit{}
		err := apiServiceAudit.LoadFromDBServiceAudit(dbServiceAudit)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiServiceAudits = append(apiServiceAudits, apiServiceAudit)
	}
	returnJSON, err := json.Marshal(apiServiceAudits)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func postServiceAudit(w http.ResponseWriter, r *http.Request) {
	var apiServiceAudit ServiceAudit

	var user string
	gu := goracleUser.GetUserFromContext(r.Context())
	if gu != nil {
		user = gu.UID
	} else {
		user = "anonymous"
	}

	defaultAction := "validated"

	// Get the parent service ID from the URL
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	defer r.Body.Close()

	// Make sure that the service at the specified ID actually exists
	_, err = catalog.GetCatalog().GetServiceByID(uint(id))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}

	err = json.NewDecoder(r.Body).Decode(&apiServiceAudit)
	if err != nil {
		logrus.Warnf("Unable to unmarshal POST service audit data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// If a service_id is given in the API request, it must match the ID
	// of the URL route
	if apiServiceAudit.ServiceID != nil && *apiServiceAudit.ServiceID != uint(id) {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("service_id must match the service in the URL route")}, 400)
	}
	// Set the ServiceID if nil
	if apiServiceAudit.ServiceID == nil {
		apiServiceAudit.ServiceID = &id
	}

	// Can't specify an ID for service audit creation
	if apiServiceAudit.ID != nil {
		err := fmt.Errorf("cannot set ID when creating a service audit")
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}
	// If no timestamp is given, use the current time
	if apiServiceAudit.AuditTime == nil {
		// Hack to get the current time as a pointer because &time.Now() doesnt work
		apiServiceAudit.AuditTime = func(t time.Time) *time.Time { return &t }(time.Now())
	}

	// Service auditor is always taken from the user, client is not trusted
	apiServiceAudit.Auditor = &user

	// If no action is given, default to "validated"
	if apiServiceAudit.Action == nil {
		apiServiceAudit.Action = &defaultAction
	}

	dbServiceAudit, err := apiServiceAudit.DumpToDBServiceAudit(&catalog.ServiceAudit{})
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	// TODO: implement
	err = catalog.GetCatalog().AddServiceAudit(dbServiceAudit)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 422)
		return
	}
	apiServiceAudit.LoadFromDBServiceAudit(dbServiceAudit)
	constructResponse(w, http.StatusCreated, apiServiceAudit)
}

// Feature
func getFeatures(w http.ResponseWriter, r *http.Request) {

	dbFeats, err := catalog.GetCatalog().GetFeatures()
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	apiFeats := make([]*Feature, 0)
	// Marshall all of the DB components into API responses
	for _, dbFeat := range dbFeats {
		apiFeat := &Feature{}
		err := apiFeat.LoadFromDBFeature(dbFeat)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiFeats = append(apiFeats, apiFeat)
	}
	returnJSON, err := json.Marshal(apiFeats)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func getFeature(w http.ResponseWriter, r *http.Request) {
	id, err := serverutil.StringToUint(pat.Param(r, "id"))
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{fmt.Errorf("invalid feature id: %s", err.Error())}, 400)
		return
	}
	dbFeat, err := catalog.GetCatalog().GetFeatureByID(id)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 404)
		return
	}
	apiFeat := &Feature{}
	err = apiFeat.LoadFromDBFeature(dbFeat)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	returnJSON, err := json.Marshal(apiFeat)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}

func getLogs(w http.ResponseWriter, r *http.Request) {
	lookupMap := make(map[string]interface{})
	offset := -1
	limit := -1
	var err error

	// lookup supported URL query params
	itemtype := r.URL.Query().Get("item_type")
	if itemtype != "" {
		lookupMap["item_type"] = itemtype
	}
	itemID := r.URL.Query().Get("item_id")
	if itemID != "" {
		lookupMap["item_id"] = itemID
	}
	action := r.URL.Query().Get("action")
	if action != "" {
		lookupMap["action"] = action
	}
	author := r.URL.Query().Get("author")
	if author != "" {
		lookupMap["author"] = author
	}
	offset_param := r.URL.Query().Get("offset")
	if offset_param != "" {
		offset, err = strconv.Atoi(offset_param)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
	}
	limit_param := r.URL.Query().Get("limit")
	if limit_param != "" {
		limit, err = strconv.Atoi(limit_param)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
	}

	dbLogs := catalog.GetCatalog().GetLogRecords(lookupMap, offset, limit)

	apiLogs := make([]*LogRecord, 0)
	// Marshall all of the DB components into API responses
	for _, dbLog := range dbLogs {
		apiLog := &LogRecord{}
		err := apiLog.LoadFromDBLogRecord(dbLog)
		if err != nil {
			serverutil.HandleAPIErrors(w, []error{err}, 500)
			return
		}
		apiLogs = append(apiLogs, apiLog)
	}
	returnJSON, err := json.Marshal(apiLogs)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}

	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusOK)
	w.Write(returnJSON)
}
