package statusserver

import (
	"github.com/sirupsen/logrus"
	_ "net/http/pprof"

	"goji.io"

	"code.justin.tv/availability/goracle/api/v2"
	"code.justin.tv/availability/goracle/config"
	"code.justin.tv/availability/goracle/guardianauth"
	"code.justin.tv/availability/goracle/messaging"
	"code.justin.tv/availability/goracle/serverutil"
	"code.justin.tv/availability/goracle/statusserver/cataloghandlers"
	"code.justin.tv/availability/goracle/statusserver/confighandlers"
	"code.justin.tv/foundation/twitchserver"
	"strings"
	"os"
	"bytes"
	"net/http"
	"encoding/json"
	"fmt"
	"net/http/httptest"
)

type statusServer struct {
	*goji.Mux
}

func (s *statusServer) addHandlers(assetPath string, requireHTTPS bool) {
	s.Use(serverutil.Logger)

	if requireHTTPS {
		s.Use(serverutil.RequireHTTPS)
	}

	guardianauth.RegisterHandlers(s.Mux)

	if strings.ToLower(os.Getenv("ENABLE_V1_API")) == "true" { //  serve /api/v1 on internal systems only
		cataloghandlers.RegisterHandlers(s.Mux)
	}
	confighandlers.RegisterHandlers(s.Mux)
	apiv2.RegisterHandlers(s.Mux)
	messaging.RegisterHandlers(s.Mux)
}

type GraphqlQuery struct {
	Operation *string                 `json:"operation"`
	Query     *string                 `json:"query"`
	Variables *map[string]interface{} `json:"variables"`
}

func simpleQuery(q string) string {
	g := GraphqlQuery{
		Query: &q,
	}
	b, err := json.Marshal(g)
	if err != nil {
		logrus.Fatal(err)
	}
	return string(b)
}

const fetchAllServices = `query FetchAllServices {
  services {
    ...serviceShallowData
    __typename
  }
}

fragment serviceShallowData on Service {
  id
  name
  state
  primary_owner {
    uid
    slack {
      id
      name
      __typename
    }
    cn
    preferred_name
    inactive
    __typename
  }
  type
  availability_objective
  component_ids
  latest_service_audits {
    id
    audit_type
    auditor
    action
    audit_time
    audit_value
    __typename
  }
  attributes {
    id
    name
    value
    __typename
  }
  __typename
}`

// RunServer sets up the UI and API routes and starts the web server
func RunServer(assetPath string) {
	server := twitchserver.NewServer()
	dimensionKey := new(int)
	elevateKey := new(int)
	s := &statusServer{server}

	s.Use(NoCache)
	s.addHandlers(assetPath, config.Config.RequireHTTPS)

	// this will add default handlers for OS signals
	twitchserver.AddDefaultSignalHandlers()
	config := &twitchserver.ServerConfig{
		DimensionKey: dimensionKey,
		ElevateKey:   elevateKey,
		Addr:         config.Config.ServerAddr,
	}
	config.RegisterPostStartupHook(func() {
		w := httptest.NewRecorder()
		req, err := http.NewRequest("POST", "http://localhost:8000/api/v2/query", bytes.NewBufferString(simpleQuery(fetchAllServices)))
		if err != nil {
			fmt.Errorf("startup hook to invoke fetchAllServices failed")
		}
		s.Mux.ServeHTTP(w, req)
		fmt.Printf("startup hook: fetchAllServices returned HTTP status %d. response body: %s", w.Code, w.Body.String())
	})

	err := twitchserver.ListenAndServe(s, config)
	if err != nil {
		logrus.Fatal(err)
	}
}
