# Creating a new environment
## What is involved?
At the top level we will be setting up the following:
- elastic beanstalk instances
- email permissions
- sandstorm permissions
- s2s permissions
- dns
- oauth permissions
- monitoring(optional)

## Setting up Beanstalk
We use the module located at the `terraform/modules/tf_beanstalk_webserver_env`
folder to do basic setup. Inside production.tf you can see an example setup.
To create a copy of that setup, you can do something like this:

```
module "new-bs-production" {
  source = "../modules/tf_beanstalk_webserver_env"
  aws_profile = "twitch-central-aws"
  eb_application_name = "${module.bs-app-production.beanstalk_application_name}"
  vpc_id = "vpc-bda8eed9"
  ec2_subnet_ids = "subnet-1a87f07e,subnet-0eb83678,subnet-828650da"
  elb_subnet_ids = "subnet-1a87f07e,subnet-0eb83678,subnet-828650da"
  elb_loadbalancer_security_groups = "sg-1825527e"
  auto_scaling_lc_security_groups = "sg-1825527e"
  solution_stack_name = "64bit Amazon Linux 2016.09 v2.3.0 running Docker 1.11.2"
  auto_scaling_lc_instance_type = "m4.xlarge"
  owner = "oetools@twitch.tv"
  service = "availability/goracle"
  env = "new-production"
  associate_public_address = "false"
  cname_prefix = "new-goracle-production"
  environment_environment_var = "production"
  asg_min_size = "4"
  asg_max_size = "8"
}
```
Pretty much everything is the same as the production.tf. What is different is:
- the module name has to be unique for the directory
- the env name has to be unique because it is used to generate the environment name
- cname_prefix has to be unique since this is used to create your cname
- environment_environment_var is used to select a set of secrets from sandstorm. This will
usually be production/development

That will setup your machines. Now remember the name of your module name,
we will refer constantly to it throughout.

## Setting up Email permissions
Goracle uses SES to handle email, make sure that the above role has access.
e.g.
```
resource "aws_iam_role_policy_attachment" "attach-email-ses-perms" {
    policy_arn = "arn:aws:iam::aws:policy/AmazonSESFullAccess"
    role       = "${module.bs-production.iam_role_id}"
}
```


## Setting up sandstorm
In order for the machines to access sandstorm secrets, the beanstalk instance
needs access to the `goracle-{environment}` roles. You can do that like so:
```
resource "aws_iam_role_policy" "sandstorm-production-newproduction" {
  name = "sandstorm-new-production-goracle-team"
  role = "${module.new-bs-production.iam_role_id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Resource": "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/goracle-production"
    },
    {
      "Sid": "Stmt1473817796000",
      "Effect": "Allow",
      "Action": "s3:GetObject",
      "Resource": "arn:aws:s3:::twitch-sandstorm/sandstorm-agent.rpm"
    }
  ]
}
EOF
}
```
You also have to add the beanstalk role manually at the dashboard
[here](https://dashboard.internal.justin.tv/sandstorm/manage-roles?role=goracle-production&action=editrole)

Once you have those, you will have access to those secrets.

## Setting up S2S
To setup S2S, you have to add the beanstalk role to the general s2s perm in s2s.tf
for example:
```
resource "aws_iam_role_policy_attachment" "attach-new-prod-goracle-s2s" {
    role        = "${module.new-bs-production.iam_role_id}"
    policy_arn  = "${aws_iam_policy.policy.arn}"
}
```
You also have to add the beanstalk role to the dashboards [here](https://dashboard.internal.justin.tv/s2s/services/YTM2ODVmZWQtNjBkOC00ODIzLWJjZjYtNjFlOGRkODJhYjg4OnNlcnZpY2UtY2F0YWxvZy1wcm9k)
Specifically, you will need to add it to allowed ARNs

## Setting up DNS
DNS is managed by [dashboard](https://dashboard.internal.justin.tv/dns/?entry=status-development.internal.justin.tv)
We have a cname that points to the beanstalk cname

## Setting up Oauth
OAuth is used by our login system and is managed through
[dashboard](https://dashboard.internal.justin.tv/guardian/)
This relies on a unique homepage with homepage redirect URL. These are linked to a client secret and ID
These are held in sandstorm for Goracle at goracle/goracle/production/oauth_client_id
and goracle/goracle/production/oauth_client_secret

## Monitoring
You can see how it is currently setup at monitoring.tf.
We create a SNS queue that is hooked up to a pagerduty integration link.
Then a cloudwatch alert is created that alerts whenever the environment is
unhealthy
