package catalog

import (
	"fmt"
	"time"

	"code.justin.tv/availability/hms-echo/pkg/jsonrequest"
)

// Org is the API visible version of catalog orgs
type Org struct {
	ID          *uint   `json:"id"`
	Name        *string `json:"name"`
	Label       *string `json:"label"`
	Description *string `json:"description"`
	Parent      *uint   `json:"parent"`
	Accounts    *[]uint `json:"accounts"`
}

// Team is the API visible version of catalog teams
type Team struct {
	ID          *uint   `json:"id"`
	Name        *string `json:"name"`
	Email       *string `json:"email"`
	Slack       *string `json:"slack"`
	Description *string `json:"description"`
	Org         *uint   `json:"org"`
}

// Service is the API visible version of catalog services
type Service struct {
	ID                    *uint      `json:"id"`
	Name                  *string    `json:"name"` // Globally unique human service name
	Description           *string    `json:"description"`
	Known                 *bool      `json:"known"`
	Type                  *string    `json:"type"`
	Environment           *string    `json:"environment"`
	Components            *[]uint    `json:"components"`
	Team                  *uint      `json:"team"`
	AvailabilityObjective *float64   `json:"availability_objective,string"`
	CreatedAt             *time.Time `json:"created"`
	UpdatedAt             *time.Time `json:"updated"`
}

// Metric is the API visible version of catalog metrics
type Metric struct {
	ID              *uint    `json:"id"`
	Label           *string  `json:"label"`
	Name            *string  `json:"name"`
	Description     *string  `json:"description"`
	AutoGenerated   *bool    `json:"autogenerated"`
	Queries         []uint   `json:"queries"`
	Threshold       *float64 `json:"threshold"`
	CalculationType *string  `json:"calculation_type"`
	ComponentRollup *bool    `json:"component_rollup"`
	// FeatureRollup   *bool    `json:"feature_rollup"`
	LatencyQuery *string `json:"latency_query"`
}

// Query is the API visible version of catalog queries
type Query struct {
	ID            *uint   `json:"id"`
	Type          *string `json:"type"`
	Query         *string `json:"query"`
	AggregateType *string `json:"aggregate_type"`
}

// ServiceAudit is the API visible version of catalog service audit actions
type ServiceAudit struct {
	ID        *uint      `json:"id"`
	ServiceID *uint      `json:"service_id"`
	AuditType *string    `json:"audit_type"`
	Auditor   *string    `json:"auditor"`
	Action    *string    `json:"action"`
	AuditTime *time.Time `json:"audit_time"`
}

type Client struct {
	Api          string
	TeamCache    []Team
	ServiceCache []Service
}

func (c *Client) FindAllQueries() ([]Query, error) {
	var allQueries []Query
	url := fmt.Sprintf("%s/queries/", c.Api)
	err := jsonrequest.Get(url, &allQueries)
	return allQueries, err
}

func (c *Client) FindAvailabilitySignoffForService(serviceID uint) ([]ServiceAudit, error) {
	var serviceAudits []ServiceAudit
	url := fmt.Sprintf("%s/services/%d/audits?type=availability&all=false", c.Api, serviceID)
	err := jsonrequest.Get(url, &serviceAudits)
	return serviceAudits, err
}

func (c *Client) FindAllOrgs() ([]Org, error) {
	var allOrgs []Org
	url := fmt.Sprintf("%s/orgs/", c.Api)
	err := jsonrequest.Get(url, &allOrgs)
	return allOrgs, err
}

func (c *Client) FindAllTeams() ([]Team, error) {
	if c.TeamCache != nil {
		return c.TeamCache, nil
	}
	var allTeams []Team
	url := fmt.Sprintf("%s/teams/", c.Api)
	err := jsonrequest.Get(url, &allTeams)
	if err != nil {
		return nil, err
	}
	c.TeamCache = allTeams
	return allTeams, err
}

func (c *Client) FindAllServices() ([]Service, error) {
	if c.ServiceCache != nil {
		return c.ServiceCache, nil
	}
	var allServices []Service
	url := fmt.Sprintf("%s/services/", c.Api)
	err := jsonrequest.Get(url, &allServices)
	if err != nil {
		return nil, err
	}
	c.ServiceCache = allServices
	return allServices, err
}

func (c *Client) FindTeamsForOrg(o Org) ([]Team, error) {
	var teams []Team
	allTeams, err := c.FindAllTeams()
	if err != nil {
		return nil, err
	}
	for _, t := range allTeams {
		if t.Org != nil && *t.Org == *o.ID {
			teams = append(teams, t)
		}
	}
	return teams, nil
}

func (c *Client) FindServicesForTeam(t Team) ([]Service, error) {
	var services []Service
	allServices, err := c.FindAllServices()
	if err != nil {
		return nil, err
	}
	for _, s := range allServices {
		if s.Team != nil && *s.Team == *t.ID {
			services = append(services, s)
		}
	}
	return services, nil
}
