package echo

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/availability/goracle/dateutil"
	c "code.justin.tv/availability/hms-echo/pkg/config"
	"code.justin.tv/availability/hms-echo/pkg/graphite"
	"code.justin.tv/availability/hms-echo/pkg/graphqlcatalog"
	"code.justin.tv/availability/hms-echo/pkg/reportbuilder"
	"code.justin.tv/availability/hms-echo/pkg/sqsbatcher"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"
)

type echoRequest struct {
	QueryID       *uint
	Query         string
	AggregateType string
	Date          time.Time
}

type sqsJob struct {
	Echo   *echoRequest
	Report *reportRequest
}

type reportRequest struct {
	ID   string
	Date time.Time
	Data interface{}
}

func sqsHandler(w http.ResponseWriter, r *http.Request) {
	logrus.Printf("sqshandler: %s %s %v", r.RemoteAddr, r.Method, r.URL)
	body, err := ioutil.ReadAll(r.Body)
	defer r.Body.Close()
	if err != nil {
		logrus.Errorf("Unable to read JSON: %s", err)
		w.WriteHeader(500)
		return
	}

	// we ignore the error here, with no header or bogus value we assume 0
	receiveCount, _ := strconv.Atoi(r.Header.Get("X-Aws-Sqsd-Receive-Count"))
	acceptMissing := false
	if receiveCount >= c.Config.AcceptAfterRetries {
		acceptMissing = true
	}

	// Handle legacy echo requests first
	var er echoRequest
	err = json.Unmarshal(body, &er)
	if err == nil {
		if er.QueryID != nil {
			if *er.QueryID == 0 {
				start := time.Now()
				err = handleUpdateAll(er)
				duration := time.Since(start)
				logrus.Warnf("Update All took %s", duration)
				if err != nil {
					logrus.Errorf("handleUpdateAll failed: %s", err)
					w.WriteHeader(500)
					return
				}
				return
			}
			err = handleEcho(er)
			if err != nil {
				logrus.Errorf("handleEcho failed: %s", err)
				w.WriteHeader(500)
				return
			}
			logrus.Infof("completed update for query %d, %s", er.QueryID, er.Date)
			w.WriteHeader(200)
			return
		}
	}

	// This should be a new sqsJob
	var job sqsJob
	err = json.Unmarshal(body, &job)

	if err != nil {
		logrus.Errorf("Unable to decode sqsJob JSON body: %s", err)
		w.WriteHeader(500)
		return
	}

	err = handleSQSJob(job, acceptMissing)
	if err != nil {
		logrus.Errorf("Unable to handleSQSJob: %s", err)
		w.WriteHeader(500)
		return
	}

	logrus.Info("Completed sqsJob", job)
	w.WriteHeader(200)
}

func handleSQSJob(job sqsJob, acceptMissing bool) error {
	if job.Echo != nil {
		return handleEcho(*job.Echo)
	} else if job.Report != nil {
		return handleReport(*job.Report, acceptMissing)
	}
	return errors.New("No job found in SQSJob")
}

func handleEcho(e echoRequest) error {
	if e.QueryID == nil {
		return fmt.Errorf("handleEcho: got nil QueryID: %v", e)
	}
	start := now.New(e.Date.UTC()).BeginningOfDay()
	end := start.Add(24 * time.Hour)
	gdata, err := graphite.RenderQuery(e.Query, start.Unix(), end.Unix())
	if err != nil {
		return fmt.Errorf("RenderQuery failed for query %d: %s", *e.QueryID, err)
	}
	if len(gdata) != 1 {
		logrus.Infof("got %d timeseries for request %d (should be 1)", len(gdata), *e.QueryID)
		return nil
	}

	af := graphite.GetAggregateFunction(graphite.AggregateType(e.AggregateType))
	bucketed := graphite.Bucketize(gdata[0], af)

	return dynamoPut(*e.QueryID, start, bucketed)

}

func FindAllQueries() ([]graphqlcatalog.Query, error) {
	qstr := `
{
  queries() {
    id
    query
    aggregate_type
    metric_id
  }
}
`
	q := graphqlcatalog.GraphqlQuery{
		Query: &qstr,
	}
	r, err := graphqlcatalog.GetGraphQL(q)
	if err != nil || r == nil || r.Data.Queries == nil {
		return nil, err
	}
	return *r.Data.Queries, nil
}
func handleUpdateAll(e echoRequest) error {
	awscfg := aws.NewConfig().WithRegion(c.Config.Region)
	if c.Config.SQSEndpoint != "" {
		awscfg.WithEndpoint(c.Config.SQSEndpoint)
	}
	batcher := sqsbatcher.SQSBatcher{
		MaxLen: 10,
		SQS:    sqs.New(session.New(), awscfg),
	}
	batcher.SetQueueUrl(c.Config.QueueURL)
	catStart := time.Now()
	queries, err := FindAllQueries()
	duration := time.Since(catStart)
	logrus.Warnf("Fetch all Queries took %s", duration)
	if err != nil {
		return err
	}
	batchStart := time.Now()
	for _, q := range queries {
		id, err := graphqlcatalog.IDStringToUint(*q.ID)
		if err != nil {
			return err
		}
		// skip dangling queries
		if q.MetricID == nil {
			continue
		}
		if q.Query == nil || *q.Query == "" || q.AggregateType == nil {
			continue
		}
		r := &echoRequest{
			QueryID:       &id,
			Query:         *q.Query,
			AggregateType: *q.AggregateType,
			Date:          e.Date,
		}
		msg, err := json.Marshal(r)
		if err != nil {
			return err
		}
		err = batcher.SendString(string(msg))
		if err != nil {
			return err
		}
	}
	duration = time.Since(batchStart)
	logrus.Warnf("Batch all Queries took %s", duration)
	return batcher.Flush()
}

func handleReport(rr reportRequest, acceptMissing bool) error {
	switch rr.ID {
	case "availability":
		return reportbuilder.AvailabilityReport(rr.Date, acceptMissing)
	case "latency":
		return reportbuilder.LatencyReport(rr.Date)
	default:
		return errors.New("Unknown report type")
	}
}

func sendSQSMessage(obj interface{}) error {
	msg, err := json.Marshal(obj)
	if err != nil {
		return err
	}
	msgbody := string(msg)

	logrus.Info("SQS Message Body:", msgbody)

	awscfg := aws.NewConfig().WithRegion(c.Config.Region)
	if c.Config.SQSEndpoint != "" {
		awscfg.WithEndpoint(c.Config.SQSEndpoint)
	}
	svc := sqs.New(session.New(), awscfg)
	in := &sqs.SendMessageInput{
		MessageBody: &msgbody,
		QueueUrl:    &c.Config.QueueURL,
	}
	_, err = svc.SendMessage(in)
	if err != nil {
		return err
	}
	return nil
}

func sendZeroQuery(t time.Time) error {
	queryID := uint(0)
	r := &echoRequest{
		QueryID: &queryID,
		Date:    t,
	}
	return sendSQSMessage(r)
}

func sendReport(id string, date time.Time, data interface{}) error {
	s := sqsJob{
		Report: &reportRequest{
			ID:   id,
			Date: date,
			Data: data,
		},
	}

	return sendSQSMessage(&s)
}

func updateAllHandler(w http.ResponseWriter, r *http.Request) {
	var err error
	logrus.Warnf("updateAllHandler: %s %s %v %v", r.RemoteAddr, r.Method, r.URL, r.Header)
	dayOffset := 0
	dayOffsetStr := r.URL.Query().Get("dayoffset")
	if dayOffsetStr != "" {
		dayOffset, err = strconv.Atoi(dayOffsetStr)
		if err != nil {
			logrus.Errorf("unparseable dayoffset: %s, %s", dayOffsetStr, err)
			w.WriteHeader(400)
		}
	}
	today := now.New(time.Now().UTC()).BeginningOfDay()
	t := today.AddDate(0, 0, dayOffset)
	err = sendZeroQuery(t)
	if err != nil {
		logrus.Errorf("sendZeroQuery failed: %s", err)
		w.WriteHeader(500)
		return
	}

	logrus.Info("completed send zero query")
	w.WriteHeader(200)
}

func reportHandler(w http.ResponseWriter, r *http.Request) {
	var err error
	logrus.Warnf("reportHandler: %s %s %v %v", r.RemoteAddr, r.Method, r.URL, r.Header)
	reportID := r.URL.Query().Get("id")
	if reportID == "" {
		logrus.Warn("Missing reportID")
		w.WriteHeader(400)
	}

	weekOffset := 0
	weekOffsetStr := r.URL.Query().Get("weekoffset")
	if weekOffsetStr != "" {
		weekOffset, err = strconv.Atoi(weekOffsetStr)
		if err != nil {
			logrus.Errorf("unparseable weekoffset: %s, %s", weekOffsetStr, err)
			w.WriteHeader(400)
		}
	}

	date := dateutil.BeginningOfISOWeek(time.Now().AddDate(0, 0, 7*weekOffset))

	err = sendReport(reportID, date, nil)
	if err != nil {
		logrus.Error("Failed to send SQS message: ", err)
		w.WriteHeader(500)
		return
	}

	logrus.Info("completed report:", reportID)
	w.WriteHeader(200)
}

func healthHandler(w http.ResponseWriter, r *http.Request) {
	logrus.Printf("healthHandler: %s %s %v", r.RemoteAddr, r.Method, r.URL)
	w.Write([]byte("OK"))
}

// Run starts the echo service
func Run() {
	http.HandleFunc("/health", healthHandler)

	// Endpoints used by cron to initiate work via SQS
	if c.Config.Environment == "production" {
		http.HandleFunc("/updateall", updateAllHandler)
	} else {
		// Disable updateall functionality in non production environments
		// as this has side effects on other systems.
		http.HandleFunc("/updateall", healthHandler)
	}
	http.HandleFunc("/report", reportHandler)

	// Handles all SQS messages
	http.HandleFunc("/", sqsHandler)


	err := http.ListenAndServe(":8000", nil)
	if err != nil {
		logrus.Fatal(err)
	}
}
