package graphite

import (
	"math"

	"github.com/sirupsen/logrus"
)

type AggregateType string

const (
	// Sum means the points will be summed
	Sum AggregateType = "sum"
	// Average means the points will be averaged
	Average = "average"
	// Max means the points will be averaged
	Max = "max"
)

// AggregateFunc is a function that combines multiple float (or nil)
// data points and returns a resulting float or nil.
type AggregateFunc func([]*float64) *float64

// GetAggregateFunction returns the proper function to use for an aggregate type
func GetAggregateFunction(aggregateType AggregateType) AggregateFunc {

	switch aggregateType {
	case Sum:
		return SumAggregate
	case Average:
		return AverageAggregate
	case Max:
		return MaxAggregate
	default:
		logrus.Info("Invalid aggregate type:", aggregateType)
		return AverageAggregate
	}

}

// SumAggregate is an aggregator that simply sums all data points and returns the result, unless
// there are no non-nil data points, in which case it returns nil.
func SumAggregate(values []*float64) *float64 {
	count := 0
	sum := 0.0
	for _, v := range values {
		if v != nil {
			sum += *v
			count++
		}
	}
	if count > 0 {
		return &sum
	}

	return nil
}

// AverageAggregate is an aggregator that simply sums all data points and returns the result, unless
func AverageAggregate(values []*float64) *float64 {
	sum := 0.0
	count := 0
	for _, v := range values {
		if v != nil {
			sum += *v
			count++
		}
	}
	if count > 0 {
		average := sum / float64(count)
		return &average
	}

	return nil
}

// MaxAggregate is an aggregator that simply max all data points and returns the result, unless
// there are no non-nil data points, in which case it returns nil.
func MaxAggregate(values []*float64) *float64 {
	count := 0
	var max float64
	for _, v := range values {
		if v != nil {
			if count == 0 {
				max = *v
			} else {
				max = math.Max(max, *v)
			}
			count++
		}
	}
	if count > 0 {
		return &max
	}

	return nil
}
