package graphite

import "time"

const BUCKET_DURATION = time.Duration(5) * time.Minute

// getBucketStart gets the start time of the bucket
func getBucket(t time.Time) uint32 {
	return uint32(t.Truncate(BUCKET_DURATION).Unix())
}

func Bucketize(gd *GraphiteData, af AggregateFunc) *GraphiteData {
	out := &GraphiteData{
		Name:   gd.Name,
		Target: gd.Target,
		Data:   make([]*GraphiteDatapoint, 0),
	}

	if len(gd.Data) == 0 {
		// FIXME: Should we return a RawData if there are no data points?
		return out
	}

	values := []*float64{}

	// FIXME: Buckets are currently timestamped with start of the bucket, not end.
	curBucket := getBucket(time.Time{})

	for _, point := range gd.Data {
		t := time.Unix(int64(*point.Timestamp), 0)
		bucket := getBucket(t)
		// crosssing into new bucket
		if bucket != curBucket {
			out.addBucket(curBucket, af, values)
			// This datapoint is in a new bucket. Clear
			// information and restart
			curBucket = bucket
			values = []*float64{}
		}
		values = append(values, point.Value)
	}
	// Handle data in the last bucket.
	out.addBucket(curBucket, af, values)
	return out
}

func (gd *GraphiteData) addBucket(bucket uint32, af AggregateFunc, values []*float64) {
	bucketVal := af(values)
	if bucketVal != nil {
		newgdp := &GraphiteDatapoint{
			Value:     bucketVal,
			Timestamp: &bucket,
		}
		gd.Data = append(gd.Data, newgdp)
	}
}
