package graphite

import (
	"math"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

func mkgraphite(name, target string, start time.Time, inc time.Duration, series []float64) *GraphiteData {
	gr := &GraphiteData{
		Name:   name,
		Target: target,
	}
	t := start
	for _, v := range series {
		tu := uint32(t.Unix())
		if !math.IsNaN(v) {
			vcopy := v // copy to get a unique pointer to each value
			gdp := &GraphiteDatapoint{&vcopy, &tu}
			gr.Data = append(gr.Data, gdp)
		}
		t = t.Add(inc)

	}
	return gr
}

func TestBucketize(t *testing.T) {
	assert := assert.New(t)
	// start time 2 miniutes before boundary
	start, err := time.Parse("2006-01-02T15:04:05", "2017-05-13T15:08:00")
	if err != nil {
		panic(err)
	}

	gd := mkgraphite("foo", "foo.bar", start, time.Minute, []float64{
		7, 9, // partial, minutes 8, 9
		1, 2, 3, 4, 5, // minutes 10-14
		1, 3, 2, 5, 4, // minutes 15-20
		1, 3, math.NaN(), math.NaN(), 2, // minutes 20-24 some missing points
		7, 9, // minutes 25-26 partial
	})

	wantstart, err := time.Parse("2006-01-02T15:04:05", "2017-05-13T15:05:00")
	if err != nil {
		panic(err)
	}

	b := Bucketize(gd, AverageAggregate)
	want := mkgraphite("foo", "foo.bar", wantstart, time.Minute*5, []float64{8, 3, 3, 2, 8})
	// yeah, we are doing equality check on floats after
	// division... happens to work for these values
	assert.Equal(want, b, "AverageAggregate doesn't match expected")

	b = Bucketize(gd, SumAggregate)
	want = mkgraphite("foo", "foo.bar", wantstart, time.Minute*5, []float64{16, 15, 15, 6, 16})
	assert.Equal(want, b, "SumAggregate doesn't match expected")

	b = Bucketize(gd, MaxAggregate)
	want = mkgraphite("foo", "foo.bar", wantstart, time.Minute*5, []float64{9, 5, 5, 3, 9})
	assert.Equal(want, b, "MaxAggregate doesn't match expected")
}
