package graphite

// Query graphite for useful things.

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"

	"github.com/sirupsen/logrus"
)

// GraphiteNode represents some bucket/leaf in the metric tree
type GraphiteNode struct {
	AllowChildren int                    `json:"allowChildren"`
	Context       map[string]interface{} `json:"context"`
	Expandable    int                    `json:"expandable"`
	ID            string                 `json:"id"`
	Leaf          int                    `json:"leaf"`
	Text          string                 `json:"text"`
}

// GraphiteData holds the results of a Graphite /render query
type GraphiteData struct {
	Name   string               `json:"name"` // an optional name to associate with the query
	Target string               `json:"target"`
	Data   []*GraphiteDatapoint `json:"datapoints"`
}

// GraphiteDatapoint holds a single datapoint
type GraphiteDatapoint struct {
	Value     *float64
	Timestamp *uint32
}

// UnmarshalJSON implements custom unmarshal logic to handle Graphite's weird JSON structure
// Adapted from: http://eagain.net/articles/go-json-array-to-struct/
func (d *GraphiteDatapoint) UnmarshalJSON(buf []byte) error {
	tmp := []interface{}{&d.Value, &d.Timestamp}
	wantLen := len(tmp)
	if err := json.Unmarshal(buf, &tmp); err != nil {
		return err
	}
	if g, e := len(tmp), wantLen; g != e {
		return fmt.Errorf("wrong number of fields in Datapoint: %d != %d", g, e)
	}
	return nil
}

// MarshalJSON implements custom unmarshal logic to handle Graphite's weird JSON structure
// Adapted from: http://eagain.net/articles/go-json-array-to-struct/
func (d *GraphiteDatapoint) MarshalJSON() ([]byte, error) {
	if d.Value == nil {
		return []byte(fmt.Sprintf("[null, %d]", *d.Timestamp)), nil
	}
	return []byte(fmt.Sprintf("[%f, %d]", *d.Value, *d.Timestamp)), nil
}

func (n *GraphiteNode) String() string {
	return n.ID + ":" + fmt.Sprintf("%d", n.Leaf) + ":" + n.Text
}

// Render -- Returns data for a given query
func RenderQuery(query string, from, until int64) ([]*GraphiteData, error) {
	query = url.QueryEscape(query)
	u := fmt.Sprintf("http://graphite-web.internal.justin.tv/render?format=json&target=%s&from=%d&until=%d", query, from, until)
	logrus.Debugln("Render:", u)
	resp, err := http.Get(u)

	if err != nil {
		logrus.Println(err)
		return nil, errors.New("query failed")
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Unexpected HTTP status: %d", resp.StatusCode)
	}
	var data []*GraphiteData

	// Render calls return an array of GraphiteData structs
	d := json.NewDecoder(resp.Body)
	err = d.Decode(&data)
	if err != nil {
		return nil, err
	}
	return data, nil
}
