// Package graphqlcatalog implements client APIs and structs to easily interact
// with our graphql catalog APIs
package graphqlcatalog

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strconv"

	c "code.justin.tv/availability/hms-echo/pkg/config"

	"github.com/sirupsen/logrus"
)

//
// These represent all of the fields that can be returned by GraphQL for
// all of the different types of objects in the catalog. They are all pointers
// so that they are all optionally filled in.
//

type Org struct {
	ID   *string `json:"id"`
	Name *string `json:"name"`
}

type Team struct {
	ID   *string `json:"id"`
	Name *string `json:"name"`
	Org  *Org    `json:"org"`
}

type Service struct {
	ID         *string      `json:"id"`
	Name       *string      `json:"name"`
	Team       *Team        `json:"team"`
	Attributes *[]Attribute `json:"attributes"`
}

type Component struct {
	ID         *string      `json:"id"`
	Name       *string      `json:"name"`
	Service    *Service     `json:"service"`
	Attributes *[]Attribute `json:"attributes"`
}

type Query struct {
	ID            *string `json:"id"`
	QueryType     *string `json:"query_type"`
	Query         *string `json:"query"`
	AggregateType *string `json:"aggregate_type"`
	MetricID      *string `json:"metric_id"`
}

type Metric struct {
	ID               *string      `json:"id"`
	Name             *string      `json:"name"`
	Component        *Component   `json:"component"`
	Queries          *[]Query     `json:"queries"`
	Attributes       *[]Attribute `json:"attributes"`
	Threshold        *float64     `json:"threshold"`
	LatencyQuery     *string      `json:"latency_query"`
	LatencyObjective *float64     `json:"latency_objective"`
}

type Attribute struct {
	Name  *string `json:"name"`
	Value *string `json:"value"`
}

// This is a "union" type with all response types we can get back. Add
// more fields as needed
type CatalogList struct {
	Metrics *[]Metric
	Queries *[]Query
}

type GraphqlQuery struct {
	Operation *string                 `json:"operation"`
	Query     *string                 `json:"query"`
	Variables *map[string]interface{} `json:"variables"`
}

type GraphQLResponse struct {
	Data CatalogList `json:"data"`
}

// GetGraphQL implements querying and returning the results of a graphQL query
func GetGraphQL(query GraphqlQuery) (*GraphQLResponse, error) {
	queryBytes, err := json.Marshal(query)
	if err != nil {
		return nil, err
	}

	client := &http.Client{}
	req, err := http.NewRequest("POST", c.Config.CatalogGraphQLAPI, bytes.NewReader(queryBytes))
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/json; charset=utf-8")
	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 && resp.StatusCode != 201 {
		return nil, fmt.Errorf("Status not OK: %d", resp.StatusCode)
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	// logrus.Println(string(body))

	if len(body) == 0 {
		// Empty data, just don't parse
		return nil, err
	}

	var r GraphQLResponse
	err = json.Unmarshal(body, &r)
	if err != nil {
		logrus.Info("Failing to unmarshal response")
		return nil, err
	}
	return &r, nil
}

func IDStringToUint(s string) (uint, error) {
	var u uint64
	var err error
	if u, err = strconv.ParseUint(s, 10, 32); err != nil {
		return 0, err
	}
	return uint(u), nil
}
