package reportbuilder

import (
	"fmt"
	"time"

	c "code.justin.tv/availability/hms-echo/pkg/config"

	"code.justin.tv/availability/goracle/dateutil"
	"code.justin.tv/availability/hms-echo/pkg/catalog"
	"code.justin.tv/availability/hms-echo/pkg/jsonrequest"
	"github.com/sirupsen/logrus"
)

type BUAvailabilityReport struct {
	ID             uint
	Name           string
	ServiceReports []ServiceAvailabilityReport
}

type ServiceAvailabilityReport struct {
	ID                    uint
	Name                  string
	TeamID                uint
	TeamName              string
	SLA                   float64
	Availability          *float64
	AvailabilityAuditDate *time.Time
}

func BuildServiceAvailabilityReport(s catalog.Service, t catalog.Team, week time.Time, cat *catalog.Client, acceptMissing bool) (*ServiceAvailabilityReport, error) {
	sr := &ServiceAvailabilityReport{
		ID:       *s.ID,
		Name:     *s.Name,
		TeamID:   *t.ID,
		TeamName: *t.Name,
		SLA:      *s.AvailabilityObjective,
	}

	//
	// Fetch availability data
	//
	start := dateutil.DateStr(week)
	end := dateutil.DateStr(week.Add(time.Duration(24*6) * time.Hour))

	type point struct {
		X time.Time `json:x`
		Y float64   `json:y`
	}

	type aData struct {
		Points []point `json:points`
	}

	url := fmt.Sprintf(c.Config.EskAPI+"/availability/summary/?service=%d&start=%s&end=%s", *s.ID, start, end)
	var resp aData
	err := jsonrequest.Get(url, &resp)
	if err != nil {
		logrus.Error("Failed to retrieve availability ", url)
		return nil, err
	}

	sum := 0.0
	count := 0.0
	for _, p := range resp.Points {
		if p.Y < 0 {
			if acceptMissing {
				logrus.Warningf("BuildServiceAvailabilityReport: acceptMissing triggered, assuming missing data to mean available for %s", *s.Name)
				sum += 100
				count++
				continue
			} else {
				// this will trigger an sqs requeue
				return nil, fmt.Errorf("BuildServiceAvailabilityReport: missing data for %s, failing run to trigger retry", *s.Name)
			}
		}
		sum += p.Y
		count++
	}
	if count > 0 {
		avg := sum / count
		sr.Availability = &avg
	}

	// Fetch latest availability service audit
	serviceAudits, err := cat.FindAvailabilitySignoffForService(*s.ID)
	if err != nil {
		return nil, err
	}

	for _, sa := range serviceAudits {
		if *sa.Action == "validated" {
			sr.AvailabilityAuditDate = sa.AuditTime
		}
	}

	return sr, nil
}
func BuildBUAvailabilityReport(org catalog.Org, week time.Time, cat *catalog.Client, acceptMissing bool) (*BUAvailabilityReport, error) {
	// reload to populate Teams
	teams, err := cat.FindTeamsForOrg(org)
	if err != nil {
		return nil, err
	}

	r := &BUAvailabilityReport{
		ID:   *org.ID,
		Name: *org.Label,
	}
	for _, team := range teams {
		services, err := cat.FindServicesForTeam(team)
		if err != nil {
			return nil, err
		}
		for _, s := range services {
			sr, err := BuildServiceAvailabilityReport(s, team, week, cat, acceptMissing)
			if err != nil {
				logrus.Warningf("Failed to get data for %s: %s", s.Name, err)
				continue
			}
			r.ServiceReports = append(r.ServiceReports, *sr)
		}
	}
	return r, nil

}

func BuildAvailabilityReport(week time.Time, acceptMissing bool) ([]BUAvailabilityReport, error) {
	bureports := make([]BUAvailabilityReport, 0)

	cat := &catalog.Client{Api: c.Config.CatalogAPI}
	orgs, err := cat.FindAllOrgs()
	if err != nil {
		return nil, err
	}

	for _, o := range orgs {
		r, err := BuildBUAvailabilityReport(o, week, cat, acceptMissing)
		if err != nil {
			return nil, fmt.Errorf("BuildBUReport failed: %s", err)
		}

		bureports = append(bureports, *r)
	}
	return bureports, nil
}
