package reportbuilder

import (
	"fmt"
	"strings"
	"time"

	c "code.justin.tv/availability/hms-echo/pkg/config"

	"code.justin.tv/availability/hms-echo/pkg/graphqlcatalog"
	"code.justin.tv/availability/hms-echo/pkg/jsonrequest"

	"github.com/sirupsen/logrus"
)

type MetricMetadata struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type ServiceMetadata struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type TeamMetadata struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type OrgMetadata struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type LatencyReportRow struct {
	Metric           MetricMetadata   `json:"metric"`
	Service          *ServiceMetadata `json:"service"`
	Team             *TeamMetadata    `json:"team"`
	Org              *OrgMetadata     `json:"org"`
	Latency          float64          `json:"latency"`
	LatencyType      string           `json:"latency_type"`
	LatencyThreshold float64          `json:"latency_threshold"`
	LatencyOKRatio   float64          `json:"latency_ok_ratio"`
	LatencyObjective float64          `json:"latency_objective"`
	Error            *string          `json:"error"`
}

type availResp struct {
	Points []struct {
		X string  `json:"x"`
		Y float64 `json:"y"`
	} `json:"points"`
}

func MetricLatencyOKRatio(avl availResp) *float64 {

	// return nil if no results to avoid division by zero
	if len(avl.Points) == 0 {
		return nil
	}

	// values are always 0 or 1 so we can just sum and take average to
	sum := 0.0
	for _, p := range avl.Points {
		sum += p.Y
	}
	rate := sum / float64(len(avl.Points))

	return &rate
}

type point struct {
	Time  time.Time `json:"time"`
	Value float64   `json:"value"`
}

type latencyResponse struct {
	P50Weeks *[]point `json:"p50_weeks"`
	P90Weeks *[]point `json:"p90_weeks"`
	P99Weeks *[]point `json:"p99_weeks"`
}

func BuildLatencyReport(week time.Time) (*[]*LatencyReportRow, error) {
	// Use GraphQL to fetch all metrics with the latencyreport attribute
	qStr := `{
    metrics(attribute_name:"latency_report") {
      id
      name
      component {
        id
        name
        service {
          id
          name
          team {
            id
            name
            org {
              id
              name
            }
          }
        }
      }
      queries {
        id
        query_type
      }
      attributes {
        name
        value
      }
      threshold
      latency_query
      latency_objective
    }
  }`

	q := graphqlcatalog.GraphqlQuery{
		Query: &qStr,
	}

	r, err := graphqlcatalog.GetGraphQL(q)
	if err != nil {
		return nil, err
	}

	latencyRows := []*LatencyReportRow{}

	// Iterate through the graphQL response and get lists of metrics and queries
	for _, m := range *r.Data.Metrics {
		if m.Component == nil {
			logrus.Infof("latencyreport: metric %d has nil component, skipping", m.ID)
			continue
		}
		comp := *m.Component
		lr := LatencyReportRow{
			Metric: MetricMetadata{
				ID:   *m.ID,
				Name: *m.Name,
			},
			LatencyThreshold: *m.Threshold,
			LatencyType:      *m.LatencyQuery,
			LatencyObjective: *m.LatencyObjective,
		}

		// we track these separately because an empty
		// attribute value is a valid way to turn latency
		// reprots on
		latencyEnabled := false
		latencyAttribute := ""
		// Iterate through attributes to find which query we want
		for _, a := range *m.Attributes {
			if strings.ToLower(*a.Name) == "latency_report" {
				latencyEnabled = true
				latencyAttribute = *a.Value
			}
		}

		if latencyEnabled {
			var resp latencyResponse
			// validate the old form of attribute values
			if strings.HasPrefix(latencyAttribute, "percentile_") && latencyAttribute != lr.LatencyType {
				errString := fmt.Sprintf("Metric %s has non-matching latency_type=%s and latency_report=%s", *m.ID, lr.LatencyType, latencyAttribute)
				logrus.Warn(errString)
				lr.Error = &errString
			}

			// Get latency information for the metric
			// FIXME: This won't let us retrieve data for a report older than 5 weeks. Ideally we fix this to use the
			// query APIs to actually go back and get old data that we request instead of
			// just using the summary. Or change the esk api to take date ranges
			url := fmt.Sprintf(c.Config.EskAPI+"/latency/summary/?metric=%s", *m.ID)
			err := jsonrequest.Get(url, &resp)
			if err != nil {
				logrus.Error("Failed to retrieve latency summary ", url)
				return nil, err
			}

			var data *[]point
			switch strings.ToLower(lr.LatencyType) {
			case "percentile_50":
				data = resp.P50Weeks
			case "percentile_90":
				data = resp.P90Weeks
			case "percentile_99":
				data = resp.P99Weeks
			}

			if data == nil {
				errString := fmt.Sprint("No ", lr.LatencyType, " for metric ", *m.ID)
				logrus.Warn(errString)
				lr.Error = &errString
			} else {
				// Find the current week and get its latency value
				for _, w := range *data {
					if w.Time.Equal(week) {
						lr.Latency = w.Value
					}
				}
			}

			var avl availResp
			url = fmt.Sprintf(c.Config.EskAPI+"/availability/?metric=%s&start=%s&end=%s", *m.ID,
				week.Format("2006-01-02"),
				week.AddDate(0, 0, 7).Format("2006-01-02"))
			err = jsonrequest.Get(url, &avl)
			if err != nil {
				logrus.Error("Failed to retrieve latency summary ", url)
				return nil, err
			}

			okRatio := MetricLatencyOKRatio(avl)

			if okRatio != nil {
				lr.LatencyOKRatio = *okRatio
			}
		}

		// Add to rows before upcoming early exits
		latencyRows = append(latencyRows, &lr)

		//
		// Add all associated metadata
		//
		if comp.Service == nil {
			logrus.Infof("latencyreport: component %d has nil component, skipping", comp.ID)
			continue
		}
		s := *comp.Service
		lr.Service = &ServiceMetadata{
			ID:   *s.ID,
			Name: *s.Name,
		}

		if s.Team == nil {
			continue
		}

		t := *s.Team
		lr.Team = &TeamMetadata{
			ID:   *t.ID,
			Name: *t.Name,
		}

		if t.Org == nil {
			continue
		}
		o := *t.Org
		lr.Org = &OrgMetadata{
			ID:   *o.ID,
			Name: *o.Name,
		}
	}

	return &latencyRows, nil
}
