package sqsbatcher

import (
	"fmt"
	"strconv"

	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/sirupsen/logrus"
)

type SendMessageBatcher interface {
	SendMessageBatch(input *sqs.SendMessageBatchInput) (*sqs.SendMessageBatchOutput, error)
}

// SQSBatcher is a helper to queue and send batched sqs requests
type SQSBatcher struct {
	SQS    SendMessageBatcher
	MaxLen int
	sqs.SendMessageBatchInput
}

// Send adds a new entry to a batch request and sends/flushes when
// needed
//
// a nil input indicates EOF and triggers the flush of the final batch
func (s *SQSBatcher) Send(entry *sqs.SendMessageBatchRequestEntry) error {
	flush := false
	if entry == nil {
		flush = true
	} else {
		s.Entries = append(s.Entries, entry)
		if len(s.Entries) >= s.MaxLen {
			flush = true
		}
	}
	if flush && len(s.Entries) > 0 {
		out, err := s.SQS.SendMessageBatch(&s.SendMessageBatchInput)
		if err != nil {
			return err
		}
		if len(out.Failed) > 0 {
			for _, f := range out.Failed {
				logrus.Errorf("SendMessageBatch entry failed: %s", f)
			}
			return fmt.Errorf("Some SendMessageBatch entries failed (%d out of %d)", len(out.Failed), len(s.Entries))
		}
		s.Entries = s.Entries[:0]
	}
	return nil
}

// SendString adds a string to the batch, auto-filling the Id
func (s *SQSBatcher) SendString(msg string) error {
	id := strconv.Itoa(len(s.Entries))
	entry := &sqs.SendMessageBatchRequestEntry{
		MessageBody: &msg,
		Id:          &id,
	}
	return s.Send(entry)
}

func (s *SQSBatcher) Flush() error {
	return s.Send(nil)
}
