package config

import (
	"fmt"
	"os"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
)

var sandstorm_arn_template string = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/syseng-%s-hms-esk"
var secret_prefix string = "syseng/hms-esk/%s/"

func getSecret(name string) (string, error) {
	env := os.Getenv("ENVIRONMENT")

	// Config for us-west-2 for STS
	awsConfig := &aws.Config{Region: aws.String("us-west-2")}
	stsclient := sts.New(session.New(awsConfig))

	// Create stscreds.AssumeRoleProvider with a 15 minute duration
	// for the role <CROSS ACCOUNT ROLE FROM DASHBOARD HERE> in our main account
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      fmt.Sprintf(sandstorm_arn_template, env),
		Client:       stsclient,
	}

	// Create credentials and config using our Assumed Role that we
	// will use to access the main account with
	credentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(credentials)

	// Finally, create a secret manager that uses the cross-account
	// config.
	sandstormManager := manager.New(manager.Config{
		AWSConfig: awsConfig,
		TableName: "sandstorm-production",
		KeyID:     "alias/sandstorm-production",
	})

	secret, err := sandstormManager.Get(name)
	if err != nil {
		return "", err
	}
	return string(secret.Plaintext[:]), nil
}

func CatalogDBPassword() string {
	env := os.Getenv("ENVIRONMENT")
	catalogdbPasswordSecret := fmt.Sprintf(secret_prefix, env) + "catalogdb_password"
	password, err := getSecret(catalogdbPasswordSecret)
	if err != nil {
		if Config.CatalogdbPassword != "" {
			return Config.CatalogdbPassword
		} else {
			panic(err)
		}
	}
	return password
}

func CatalogDBUsername() string {
	env := os.Getenv("ENVIRONMENT")
	catalogdbUsernameSecret := fmt.Sprintf(secret_prefix, env) + "catalogdb_username"
	username, err := getSecret(catalogdbUsernameSecret)
	if err != nil {
		if Config.CatalogdbUsername != "" {
			return Config.CatalogdbUsername
		} else {
			panic(err)
		}
	}
	return username
}
