package catalog

import (
	"errors"
	"fmt"

	"github.com/jinzhu/gorm"
	"github.com/sirupsen/logrus"
)

// Component contians the information that identifies a peice of infastructure
type Component struct {
	gorm.Model
	// Description  string
	Label       string `gorm:"not null;unique"`
	Claimed     bool
	Name        string
	Description string `gorm:"size:2000"`
	Type        string
	ServiceID   uint
	Service     *Service
	Rollup      bool // This is for rolling up into a service
	Metrics     []*Metric
	AccountID   uint
}

func (c *Component) Account() *Account {
	if c.AccountID == 0 {
		return nil
	}
	acct, err := GetCatalog().GetAccountByID(c.AccountID)
	if err != nil {
		logrus.Warnf("failure getting Account data for Component: %s", c.Label)
		return nil
	}
	return acct
}

// URL returns the endpoint for the show page for a specific component
func (c *Component) URL() string {
	return "/components/" + fmt.Sprint(c.ID)
}

// BeforeSave is a gorm-invoked callback that is used for validation
func (c *Component) BeforeSave() error {
	if c.Label == "" {
		return errors.New("component label cannot be empty")
	}
	if c.Name == "" {
		return errors.New("component name cannot be empty")
	}
	if c.Type == "" {
		return errors.New("component type cannot be empty")
	}
	return nil
}

// DefaultComponent returns a zero component
func DefaultComponent() *Component {
	// Most of these wouldn't be different
	// if they were just excluded (zero values in Go),
	// but let's be explicit
	return &Component{
		Label:   getuuid(),
		Type:    "custom",
		Claimed: false,
	}
}
