package catalog

import (
	"fmt"
	"strings"
	"time"

	"github.com/sirupsen/logrus"

	"github.com/jinzhu/gorm"
	// gorm uses this internally when the engine is mysql
	_ "github.com/jinzhu/gorm/dialects/mysql"
	// gorm uses this internally when the engine is go-sqlite3
	_ "github.com/mattn/go-sqlite3"
)

type dbCatalog struct {
	dsn string
	db  *gorm.DB
}

var catalogDB dbCatalog

// SetupDB sets up the singleton db connection to the dsn passed. The caller must call CloseDB.
func SetupDB(dsn string) {
	s := strings.SplitN(dsn, "://", 2)
	if len(s) != 2 {
		logrus.Fatalf("Couldn't parse catalogDB DSN URL %s", dsn)
	}
	db, err := gorm.Open(s[0], s[1])
	if err != nil {
		logrus.Fatalf("Failed to open catalogDB connection to %s: %s", dsn, err)
	}
	logrus.Printf("Opened catalogDB connection to %s", dsn)
	catalogDB = dbCatalog{dsn: dsn, db: db.Set("gorm:save_associations", false)}
	DoMigrations(db)
}

func getDBCatalog() *dbCatalog {
	return &catalogDB
}

// CloseDB ends the singleton db connection passed. It should be called after SetupDB
func CloseDB() {
	catalogDB.db.Close()
}

func (cat *dbCatalog) GetComponentByID(id uint) (*Component, error) {
	var comp Component
	//res := cat.db.First(&comp, id)
	res := cat.db.Preload("Service").Preload("Metrics.Queries").First(&comp, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &comp, res.Error
	}
	return &comp, nil
}
func (cat *dbCatalog) GetTeamByID(id uint) (*Team, error) {
	var team Team
	res := cat.db.Preload("Org").First(&team, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &team, res.Error
	}
	return &team, nil
}

func (cat *dbCatalog) GetServiceTypeByID(id uint) (*ServiceType, error) {
	var servType ServiceType
	res := cat.db.First(&servType, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &servType, res.Error
	}
	return &servType, nil
}

func (cat *dbCatalog) GetServiceByID(id uint) (*Service, error) {
	var serv Service
	res := cat.db.Where("id = ?", id).Preload("Components").Preload("Components.Metrics.Queries").Preload("Team").Preload("Team.Org").Preload("ServiceType").First(&serv)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &serv, res.Error
	}
	return &serv, nil
}
func (cat *dbCatalog) GetFeatureByID(id uint) (*Feature, error) {
	var feat Feature
	res := cat.db.Where("id = ?", id).Preload("Metrics.Features").Preload("Metrics.Queries").Preload("Children").Preload("Children.Metrics.Queries").First(&feat)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &feat, res.Error
	}
	return &feat, nil
}
func (cat *dbCatalog) GetMetricByID(id uint) (*Metric, error) {
	var met Metric
	res := cat.db.Preload("Component").Preload("Features").Preload("Queries").First(&met, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &met, res.Error
	}
	return &met, nil
}
func (cat *dbCatalog) GetQueryByID(id uint) (*Query, error) {
	var q Query
	res := cat.db.Where("id = ?", id).First(&q, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &q, res.Error
	}
	return &q, nil
}
func (cat *dbCatalog) GetAccountByID(id uint) (*Account, error) {
	var a Account
	res := cat.db.Where("id = ?", id).Preload("Components").Preload("Components.Metrics").Preload("Components.Metrics.Queries").First(&a, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &a, res.Error
	}
	return &a, nil
}
func (cat *dbCatalog) GetOrgByID(id uint) (*Org, error) {
	var o Org
	res := cat.db.Where("id = ?", id).Preload("Teams").Preload("Accounts").First(&o, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &o, res.Error
	}
	return &o, nil
}
func (cat *dbCatalog) GetServiceDependencyByID(id uint) (*ServiceDependency, error) {
	var sd ServiceDependency
	res := cat.db.Where("id = ?", id).First(&sd, id)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return &sd, res.Error
	}
	return &sd, nil
}
func (cat *dbCatalog) GetComponents(query map[string]interface{}) ([]*Component, error) {
	return cat.GetComponentsComplete(nil, query)
}
func (cat *dbCatalog) GetComponentsByIDs(ids []uint) ([]*Component, error) {
	return cat.GetComponentsComplete(ids, nil)
}
func (cat *dbCatalog) GetComponentsComplete(ids []uint, query map[string]interface{}) ([]*Component, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var comps []*Component
	res := request.Preload("Service.Components").Preload("Service.Team").Preload("Service.ServiceType").Preload("Metrics").Preload("Metrics.Queries").Find(&comps)
	if res.Error != nil {
		return nil, res.Error
	}
	return comps, nil
}
func (cat *dbCatalog) GetServices(query map[string]interface{}) ([]*Service, error) {
	return cat.GetServicesComplete(nil, query)
}
func (cat *dbCatalog) GetServicesByIDs(ids []uint) ([]*Service, error) {
	return cat.GetServicesComplete(ids, nil)
}
func (cat *dbCatalog) GetServicesComplete(ids []uint, query map[string]interface{}) ([]*Service, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var servs []*Service
	res := request.Preload("ServiceAudits").Preload("Components").Preload("Components.Metrics.Queries").Preload("Team").Preload("Team.Org").Preload("ServiceType").Find(&servs)
	if res.Error != nil {
		return nil, res.Error
	}
	return servs, nil
}
func (cat *dbCatalog) GetTeams() []*Team {
	teams, _ := cat.GetTeamsComplete(nil, nil)
	return teams
}
func (cat *dbCatalog) GetTeamsByIDs(ids []uint) ([]*Team, error) {
	return cat.GetTeamsComplete(ids, nil)
}
func (cat *dbCatalog) GetTeamsComplete(ids []uint, query map[string]interface{}) ([]*Team, error) {
	if query == nil {
		query = make(map[string]interface{})
	}
	var teams []*Team
	res := cat.db.Where(query).Preload("Org").Find(&teams)
	if res.Error != nil {
		return nil, res.Error
	}
	return teams, nil
}
func (cat *dbCatalog) GetServiceTypes() ([]*ServiceType, error) {
	var servTypes []*ServiceType
	res := cat.db.Find(&servTypes)
	if res.Error != nil {
		return nil, res.Error
	}
	return servTypes, nil
}
func (cat *dbCatalog) GetFeatures() ([]*Feature, error) {
	return cat.GetFeaturesComplete(nil, nil)
}
func (cat *dbCatalog) GetFeaturesByIDs(ids []uint) ([]*Feature, error) {
	return cat.GetFeaturesComplete(ids, nil)
}
func (cat *dbCatalog) GetFeaturesComplete(ids []uint, query map[string]interface{}) ([]*Feature, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var feats []*Feature
	res := request.Preload("Children.Metrics.Queries").Preload("Metrics.Features").Preload("Metrics.Queries").Preload("Children").Find(&feats)
	if res.Error != nil {
		return nil, res.Error
	}
	return feats, nil
}
func (cat *dbCatalog) GetMetrics(query map[string]interface{}) []*Metric {
	mets, _ := cat.GetMetricsComplete(nil, query)
	return mets
}
func (cat *dbCatalog) GetMetricsByIDs(ids []uint) ([]*Metric, error) {
	return cat.GetMetricsComplete(ids, nil)
}
func (cat *dbCatalog) GetMetricsComplete(ids []uint, query map[string]interface{}) ([]*Metric, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var mets []*Metric
	res := request.Preload("Component").Preload("Queries").Find(&mets)
	if res.Error != nil {
		return nil, res.Error
	}
	return mets, nil
}
func (cat *dbCatalog) GetQueries(query map[string]interface{}) []*Query {
	queries, _ := cat.GetQueriesComplete(nil, query)
	return queries
}
func (cat *dbCatalog) GetQueriesByIDs(ids []uint) ([]*Query, error) {
	return cat.GetQueriesComplete(ids, nil)
}
func (cat *dbCatalog) GetQueriesComplete(ids []uint, query map[string]interface{}) ([]*Query, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var queries []*Query
	res := request.Find(&queries)
	if res.Error != nil {
		return nil, res.Error
	}
	return queries, nil
}
func (cat *dbCatalog) GetAccounts(query map[string]interface{}) []*Account {
	acts, _ := cat.GetAccountsComplete(nil, query)
	return acts
}
func (cat *dbCatalog) GetAccountsByIDs(ids []uint) ([]*Account, error) {
	return cat.GetAccountsComplete(ids, nil)
}
func (cat *dbCatalog) GetAccountsComplete(ids []uint, query map[string]interface{}) ([]*Account, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var accounts []*Account
	res := request.Preload("Components").Find(&accounts)
	if res.Error != nil {
		return nil, res.Error
	}
	return accounts, nil
}
func (cat *dbCatalog) GetOrgs(query map[string]interface{}) []*Org {
	orgs, _ := cat.GetOrgsComplete(nil, query)
	return orgs
}
func (cat *dbCatalog) GetOrgsByIDs(ids []uint) ([]*Org, error) {
	return cat.GetOrgsComplete(ids, nil)
}
func (cat *dbCatalog) GetOrgsComplete(ids []uint, query map[string]interface{}) ([]*Org, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var orgs []*Org
	res := request.Preload("Accounts").Preload("Teams").Where(query).Find(&orgs)
	if res.Error != nil {
		return nil, res.Error
	}
	return orgs, nil
}

func (cat *dbCatalog) GetServiceDependencies(query map[string]interface{}) ([]*ServiceDependency, error) {
	return cat.GetServiceDependenciesComplete(nil, query)
}
func (cat *dbCatalog) GetServiceDependenciesByIDs(ids []uint) ([]*ServiceDependency, error) {
	return cat.GetServiceDependenciesComplete(ids, nil)
}
func (cat *dbCatalog) GetServiceDependenciesComplete(ids []uint, query map[string]interface{}) ([]*ServiceDependency, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var servDeps []*ServiceDependency
	res := request.Where(query).Find(&servDeps)
	if res.Error != nil {
		return nil, res.Error
	}
	return servDeps, nil
}

// GetLatestServiceAudits fetches, for each valid audit type, the
// latest audit of that type on the specified service
func (cat *dbCatalog) GetLatestServiceAudits(query map[string]interface{}) []*ServiceAudit {
	if query == nil {
		query = make(map[string]interface{})
	}
	typesToFetch := ServiceAuditTypes
	// Override type targets if the query map wants to filter on it
	if t, ok := query["audit_type"]; ok {
		typesToFetch = []string{t.(string)}
	}

	var serviceAudits []*ServiceAudit
	// Do one query against each audit type we care about
	for _, auditType := range typesToFetch {
		query["audit_type"] = auditType
		tmpResults := []*ServiceAudit{}
		cat.db.Where(query).Order("audit_time desc").Limit(1).Find(&tmpResults)
		if len(tmpResults) > 1 {
			logrus.Warnf("multiple 'latest' service audits found for service=%d and audit_type=%d",
				query["service_id"].(uint),
				auditType,
			)
		}
		serviceAudits = append(serviceAudits, tmpResults...)
	}
	return serviceAudits
}

func (cat *dbCatalog) GetServiceAuditTypes() []string {
	return ServiceAuditTypes
}

// GetServiceAudits fetches service audits. Usually called with a
// service_id filter
func (cat *dbCatalog) GetServiceAudits(query map[string]interface{}) []*ServiceAudit {
	if query == nil {
		query = make(map[string]interface{})
	}
	var serviceAudits []*ServiceAudit
	cat.db.Where(query).Order("audit_time desc").Find(&serviceAudits)
	return serviceAudits
}

// GetLogRecords fetches change log records that satisfy the query
func (cat *dbCatalog) GetLogRecords(query map[string]interface{}, offset, limit int) []*LogRecord {
	if query == nil {
		query = make(map[string]interface{})
	}
	var logRecords []*LogRecord
	cat.db.Where(query).Offset(offset).Limit(limit).Order("id desc").Find(&logRecords)
	return logRecords
}

func (cat *dbCatalog) GetRootFeatures() []*Feature {
	var feat Feature
	cat.db.Where("features.feature_id IS NULL").Preload("Children.Metrics.Queries").Preload("Metrics.Features").Preload("Metrics.Queries").Preload("Children").First(&feat)

	return []*Feature{&feat}
}
func (cat *dbCatalog) AddComponent(c *Component) error {
	tx := cat.db.Begin()
	err := tx.Save(c).Error
	if err != nil {
		tx.Rollback()
		return err
	}

	if c.ID != 0 {
		oldMetrics := []*Metric{}
		tx.Model(c).Related(&oldMetrics)
		for _, oldMetric := range oldMetrics {
			oldMetric.ComponentID = 0
			tx.Save(oldMetric)
		}
		for _, m := range c.Metrics {
			m.ComponentID = c.ID
			tx.Save(m)
		}
	}
	return tx.Commit().Error
}

func (cat *dbCatalog) AddService(s *Service) error {
	tx := cat.db.Begin()
	err := tx.Save(s).Error
	if err != nil {
		tx.Rollback()
		return err
	}

	if s.ID != 0 {
		// correct the Service for each component
		oldComponents := []*Component{}
		tx.Model(s).Related(&oldComponents)
		for _, oldComp := range oldComponents {
			oldComp.ServiceID = 0
			tx.Save(oldComp)
		}
		for _, c := range s.Components {
			c.ServiceID = s.ID
			tx.Save(c)
		}
	}
	return tx.Commit().Error
}

// THIS SHOULD GO AWAY ASAP
func (cat *dbCatalog) AddServiceType(servType *ServiceType) error {
	return cat.db.Save(servType).Error
}
func (cat *dbCatalog) AddTeam(team *Team) error {
	return cat.db.Save(team).Error
}
func (cat *dbCatalog) AddFeature(feat *Feature) error {
	return cat.db.Save(feat).Error
}
func (cat *dbCatalog) AddMetric(met *Metric) error {
	tx := cat.db.Begin()
	err := tx.Save(met).Error
	if err != nil {
		tx.Rollback()
		return err
	}

	if met.ID != 0 {
		oldQueries := []*Query{}
		tx.Model(met).Related(&oldQueries)
		for _, oldQuery := range oldQueries {
			oldQuery.MetricID = 0
			tx.Save(oldQuery)
		}
		for _, q := range met.Queries {
			q.MetricID = met.ID
			tx.Save(q)
		}
	}
	return tx.Commit().Error
}

func (cat *dbCatalog) AddQuery(q *Query) error {
	return cat.db.Save(q).Error
}

func (cat *dbCatalog) AddAccount(a *Account) error {
	return cat.db.Save(a).Error
}

func (cat *dbCatalog) AddOrg(o *Org) error {
	tx := cat.db.Begin()
	err := tx.Save(o).Error
	if err != nil {
		tx.Rollback()
		return err
	}

	if o.ID != 0 {
		tx.Model(o).Association("Accounts").Replace(o.Accounts)
	}
	return tx.Commit().Error
}

// AddServiceAudit writes a service audit log into the database
func (cat *dbCatalog) AddServiceAudit(sa *ServiceAudit) error {
	return cat.db.Save(sa).Error
}

// AddLogRecord writes a change log record into the database
func (cat *dbCatalog) AddLogRecord(lr *LogRecord) error {
	return cat.db.Save(lr).Error
}

// AddServiceDependency Creates a new service dependency in the database
func (cat *dbCatalog) AddServiceDependency(sd *ServiceDependency) error {
	return cat.db.Save(sd).Error
}

func (cat *dbCatalog) DeleteService(s *Service) error {
	if s.ID == 0 {
		logrus.Warnf("refusing to delete service with unset ID!")
		return nil
	}
	tx := cat.db.Begin()
	// Clear associations by resetting component service_ids to 0
	oldComponents := []*Component{}
	tx.Model(s).Related(&oldComponents)
	for _, oldComp := range oldComponents {
		oldComp.ServiceID = 0
		tx.Save(oldComp)
	}
	t := time.Now()
	tx.Model(s).Update("name", fmt.Sprint(s.Name, t.Format("2006-01-02 15:04:05")))
	tx.Delete(s)
	return tx.Commit().Error
}
func (cat *dbCatalog) DeleteComponent(c *Component) error {
	if c.ID == 0 {
		logrus.Warnf("refusing to delete component with unset ID!")
		return nil
	}
	tx := cat.db.Begin()
	oldMetrics := []*Metric{}
	tx.Model(c).Related(&oldMetrics)
	for _, oldMetric := range oldMetrics {
		oldMetric.ComponentID = 0
		tx.Save(oldMetric)
	}
	tx.Delete(c)
	return tx.Commit().Error
}
func (cat *dbCatalog) DeleteMetric(m *Metric) error {
	if m.ID == 0 {
		return fmt.Errorf("Zero metric ID in Delete")
	}
	tx := cat.db.Begin()
	oldQueries := []*Query{}
	tx.Model(m).Related(&oldQueries)
	for _, oldQuery := range oldQueries {
		oldQuery.MetricID = 0
		tx.Save(oldQuery)
	}
	// This use of Clear() is fine because its maintained using
	// a many-to-many table, so gorm should just remove the row
	tx.Model(m).Association("Features").Clear()
	tx.Delete(m)
	return tx.Commit().Error
}

func (cat *dbCatalog) DeleteTeam(t *Team) error {
	if t.ID == 0 {
		return fmt.Errorf("Zero team ID in Delete")
	}
	services, err := cat.GetServices(map[string]interface{}{"team_id": t.ID})
	if err != nil {
		return fmt.Errorf("Couldn't get services: %s", err)
	}
	for _, s := range services {
		return fmt.Errorf("Team is is in use by service %s with ID %d", s.Name, s.ID)
	}
	return cat.db.Delete(t).Error
}
func (cat *dbCatalog) DeleteQuery(q *Query) error {
	if q.ID == 0 {
		return fmt.Errorf("Zero query ID in Delete")
	}
	return cat.db.Delete(q).Error
}
func (cat *dbCatalog) DeleteAccount(a *Account) error {
	if a.ID == 0 {
		return fmt.Errorf("Zero account ID in Delete")
	}
	tx := cat.db.Begin()
	oldComponents := []*Component{}
	tx.Model(a).Related(&oldComponents)
	for _, oldComponent := range oldComponents {
		oldComponent.AccountID = 0
		tx.Save(oldComponent)
	}
	t := time.Now()
	cat.db.Model(a).Update("aws_account_id", fmt.Sprint(a.AWSAccountID, t.Format("2006-01-02 15:04:05")))
	tx.Delete(a)
	return tx.Commit().Error
}
func (cat *dbCatalog) DeleteOrg(o *Org) error {
	if o.ID == 0 {
		return fmt.Errorf("Zero org ID in Delete")
	}
	tx := cat.db.Begin()
	oldAccounts := []*Account{}
	tx.Model(o).Related(&oldAccounts)
	for _, oldAccount := range oldAccounts {
		oldAccount.OrgID = 0
		tx.Save(oldAccount)
	}
	oldTeams := []*Team{}
	tx.Model(o).Related(&oldTeams)
	for _, oldTeam := range oldTeams {
		oldTeam.OrgID = 0
		tx.Save(oldTeam)
	}
	t := time.Now()
	tx.Model(o).Update("name", fmt.Sprint(o.Name, t.Format("2006-01-02 15:04:05")))
	tx.Delete(o)
	return tx.Commit().Error
}
func (cat *dbCatalog) DeleteServiceDependency(sd *ServiceDependency) error {
	if sd.ID == 0 {
		logrus.Warnf("refusing to delete service dependency with unset ID!")
		return nil
	}
	return cat.db.Delete(sd).Error
}

func (cat *dbCatalog) GetServicesByType(servType ServiceType) []*Service {
	var servs []*Service
	cat.db.Where("service_type_id = ?", servType.ID).Preload("ServiceAudits").Preload("Components.Metrics.Queries").Preload("Team").Preload("Team.Org").Find(&servs)
	return servs
}
func (cat *dbCatalog) UnknownComponents() []*Component {
	var comps []*Component
	cat.db.Where("service_id = ?", 0).Preload("Metrics.Queries").Find(&comps)
	return comps
}

func (cat *dbCatalog) GetAttributes(query map[string]interface{}) ([]*Attribute, error) {
	return cat.GetAttributesComplete(nil, query)
}
func (cat *dbCatalog) GetAttributesByIDs(ids []uint) ([]*Attribute, error) {
	return cat.GetAttributesComplete(ids, nil)
}
func (cat *dbCatalog) GetAttributesComplete(ids []uint, query map[string]interface{}) ([]*Attribute, error) {
	request := cat.db
	if ids != nil {
		request = request.Where(ids)
	}
	if query != nil {
		request = request.Where(query)
	}
	var attributes []*Attribute
	res := request.Find(&attributes)
	if res.Error != nil {
		return nil, res.Error
	}
	return attributes, nil
}

func (cat *dbCatalog) AddAttribute(a *Attribute) error {
	return cat.db.Save(a).Error
}

func (cat *dbCatalog) GetAttributeByID(id uint) (*Attribute, error) {
	var attribute Attribute
	res := cat.db.Where("id = ?", id).First(&attribute)
	if res.Error != nil {
		// TODO: CHANGE TO NIL
		return nil, res.Error
	}
	return &attribute, nil
}

func (cat *dbCatalog) DeleteAttribute(a *Attribute) error {
	if a.ID == 0 {
		return fmt.Errorf("Zero attribute ID in Delete")
	}
	return cat.db.Unscoped().Delete(a).Error
}
