package catalog

import (
	"encoding/json"

	"github.com/jinzhu/gorm"
	"github.com/sirupsen/logrus"
)

const (
	LogTypeService           = "service"
	LogTypeComponent         = "component"
	LogTypeMetric            = "metric"
	LogTypeQuery             = "query"
	LogTypeTeam              = "team"
	LogTypeAccount           = "account"
	LogTypeOrg               = "org"
	LogTypeAttribute         = "attribute"
	LogTypeFeature           = "feature"
	LogTypeServiceDependency = "service_dependency"

	LogOpCreate = "create"
	LogOpUpdate = "update"
	LogOpDelete = "delete"
)

// LogRecord is used for audit log trails
type LogRecord struct {
	gorm.Model
	Action    string
	ItemType  string
	ItemLabel string
	ItemID    uint
	Before    string `gorm:"size:4000"`
	After     string `gorm:"size:4000"`
	Author    string
}

// WithBefore sets the Before field of a LogRecord
func (l *LogRecord) WithBefore(r interface{}) *LogRecord {
	j, err := json.Marshal(r)
	if err != nil {
		logrus.Errorf("Failed to encode json for LogRecord.WithBefore: %s", err)
		return l
	}
	l.Before = string(j)
	return l

}

// WithAfter sets the After field of a LogRecord
func (l *LogRecord) WithAfter(r interface{}) *LogRecord {
	j, err := json.Marshal(r)
	if err != nil {
		logrus.Errorf("Failed to encode json for LogRecord.WithBefore: %s", err)
		return l
	}
	l.After = string(j)
	return l
}

type LogInfo struct {
	Type  string
	ID    uint
	Label string
	Data  string
}

type Loggable interface {
	LogInfo() *LogInfo
}

func NewLogRecordFromLogInfos(action string, bl, al *LogInfo, author string) *LogRecord {
	lr := &LogRecord{
		Action: action,
		Author: author,
	}
	if bl == nil && al == nil {
		logrus.Warningf("NewLogRecordFromLogInfo received nil for both before and after parameters, this will result in an unhelpful log entry: %+v", lr)
	}
	if bl != nil {
		lr.ItemType = bl.Type
		lr.ItemID = bl.ID
		lr.ItemLabel = bl.Label
		lr.Before = bl.Data
	}
	if al != nil {
		// when both before and after are passed the after fields take precedence
		lr.ItemType = al.Type
		lr.ItemID = al.ID
		lr.ItemLabel = al.Label
		lr.After = al.Data
	}
	return lr
}
