package catalog

import (
	"time"

	"github.com/jinzhu/gorm"
	"github.com/sirupsen/logrus"
	gormigrate "gopkg.in/gormigrate.v1"
)

var serviceTypeQueries = []string{
	`REPLACE INTO service_types VALUES(1,'2017-03-29 19:08:04','2017-04-06 21:24:19',NULL,'Internal Operational Service','internal-ops','Internal only service that support operations, but not the product directly. e.g. DNS, proxy');`,
	`REPLACE INTO service_types VALUES(2,'2017-03-29 19:08:04','2017-03-31 22:29:36',NULL,'External Product Service','external-product','Externally facing services that support the Twitch product.');`,
	`REPLACE INTO service_types VALUES(3,'2017-03-29 19:08:04','2017-04-06 21:01:21',NULL,'Internal Product Service','internal-product','Internal-only service that supports the Twitch product.');`,
}

func DoMigrations(db *gorm.DB) {
	logrus.Info("Running migrations if necessary:")
	m := gormigrate.New(db, gormigrate.DefaultOptions, []*gormigrate.Migration{
		{
			ID: "201704281355",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Initial migration")
				// Preserve the specific struct so there are no side effects as we continue migrating.
				type Component struct {
					gorm.Model
					// Description  string
					Label       string `gorm:"not null;unique"`
					Claimed     bool
					Name        string
					Description string `gorm:"size:2000"`
					Type        string
					ServiceID   uint
					Rollup      bool // This is for rolling up into a service
					Metrics     []*Metric
				}
				type Feature struct {
					gorm.Model
					Label       string
					Name        string
					Description string `gorm:"size:2000"`

					Metrics   []*Metric `gorm:"many2many:feature_metric;"`
					Children  []*Feature
					FeatureID *uint //this is the ParentID but gorm requires it to be called this
					Feature   *Feature
				}

				type Metric struct {
					gorm.Model
					Label           string
					Name            string
					Description     string     `gorm:"size:2000"`
					Features        []*Feature `gorm:"many2many:feature_metric;"`
					FeatureRollup   bool       // This is only for feature Rollups
					ComponentRollup bool       // This is only for component Rollups
					AutoGenerated   bool
					Queries         []*Query
					Threshold       float64
					ComponentID     uint
					LatencyQuery    QueryType
					CalculationType CalculationType
				}

				type Query struct {
					gorm.Model
					QueryType QueryType
					MetricID  uint
					// DataSource DataSource
					Query         string `gorm:"type:varchar(1000)"`
					AggregateType AggregateType
				}

				type Service struct {
					gorm.Model
					Name          string `gorm:"not null;unique"` // Globally unique human service name
					Description   string `gorm:"size:2000"`
					TeamID        uint   // Service owner ID
					Known         bool
					ServiceTypeID uint
					ServiceType   *ServiceType
					Environment   string
					Components    []*Component
					Team          *Team
				}

				type Team struct {
					gorm.Model
					Name        string
					Label       string
					Email       string
					Slack       string
					Description string `gorm:"size:2000"`
				}

				type ServiceType struct {
					gorm.Model
					Name        string
					Label       string
					Description string `gorm:"size:2000"`
				}

				return tx.AutoMigrate(&Service{}, &Team{}, &ServiceType{}, &Component{}, &Query{}, &Feature{}, &Metric{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},
		{
			ID: "201705011009",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Default service types")
				for _, q := range serviceTypeQueries {
					if err := tx.Exec(q).Error; err != nil {
						return err
					}
				}
				return nil
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},
		{
			ID: "201705020235",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Accounts and Orgs")

				// Account represents AWS accounts
				type Account struct {
					gorm.Model
					Alias        string `gorm:"size:64"`
					Description  string `gorm:"size:2000"`
					AWSAccountID string `gorm:"not null;unique"`
					OrgID        uint
					Org          *Org
					Components   []*Component
				}

				// Org represents a Twitch organization.
				type Org struct {
					gorm.Model
					Name        string `gorm:"not null;unique"`
					Description string `gorm:"size:2000"`
					OrgID       uint   // This is the Parent but gorm requires it to be called this
					Accounts    []*Account
				}

				// Add new foreign keys
				type Component struct {
					gorm.Model
					AccountID uint
					Account   *Account
				}
				type Team struct {
					gorm.Model
					OrgID uint
					Org   *Org
				}
				return tx.AutoMigrate(&Account{}, &Org{}, &Component{}, &Team{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				// FIXME: Check errors
				tx.DropTable("accounts")
				tx.DropTable("orgs")
				tx.Table("components").DropColumn("account_id")
				tx.Table("teams").DropColumn("org_id")
				return nil
			},
		},
		{
			ID: "201705081407",
			Migrate: func(tx *gorm.DB) error {
				logrus.Warn("Migration: Org label field")
				// Org represents a Twitch organization.
				type Org struct {
					gorm.Model
					Label string
				}
				return tx.AutoMigrate(&Org{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return tx.Table("orgs").DropColumn("label").Error
			},
		},
		{
			ID: "201705191006",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Service Audits")
				type ServiceAudit struct {
					gorm.Model
					ServiceID uint
					AuditType string
					Auditor   string
					AuditTime time.Time
				}
				return tx.AutoMigrate(&ServiceAudit{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},
		{
			ID: "201705251458",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Service Audit Action field")
				type ServiceAudit struct {
					gorm.Model
					Action string
				}
				return tx.AutoMigrate(&ServiceAudit{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return tx.Table("service_audits").DropColumn("action").Error
			},
		},
		{
			ID: "201705311400",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Service Availability Objective field")
				type Service struct {
					gorm.Model
					AvailabilityObjective float64 `sql:"default:99.9"`
				}
				return tx.AutoMigrate(&Service{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return tx.Table("service").DropColumn("availability_objective").Error
			},
		},

		{
			ID: "201706261201",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: LogRecord")
				type LogRecord struct {
					gorm.Model
					Action    string
					ItemType  string
					ItemLabel string
					ItemID    uint
					Before    string `gorm:"size:4000"`
					After     string `gorm:"size:4000"`
					Author    string
				}
				return tx.AutoMigrate(&LogRecord{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},

		{
			ID: "201707111447",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Attribute")
				type Attribute struct {
					gorm.Model
					ObjectType string `gorm:"not null;unique_index:name_object_id_type"`
					ObjectId   uint   `gorm:"not null;unique_index:name_object_id_type"`
					Name       string `gorm:"not null;unique_index:name_object_id_type"`
					Value      string `gorm:"size:2000"`
				}
				return tx.AutoMigrate(&Attribute{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},
		{
			ID: "201707311606",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: ServiceDependency")
				type ServiceDependency struct {
					gorm.Model
					RootServiceID       uint
					DownstreamServiceID uint
				}
				return tx.AutoMigrate(&ServiceDependency{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},
		{
			ID: "201709151250",
			Migrate: func(tx *gorm.DB) error {
				logrus.Info("Migration: Metric Add Latency Objective")
				type Metric struct {
					gorm.Model
					LatencyObjective float64 `sql:"default:99.9"`
				}
				return tx.AutoMigrate(&Metric{}).Error
			},
			Rollback: func(tx *gorm.DB) error {
				return nil
			},
		},
	})

	if err := m.Migrate(); err != nil {
		logrus.Fatalf("Could not migrate: %v", err)
	}
	logrus.Info("Running migrations: complete")
}
