package catalog

import (
	"database/sql/driver"

	"github.com/jinzhu/gorm"
)

// QueryType defines how a query should be treated in calculations
type QueryType string

const (
	// ErrorCountQuery is a query that should return a count of errors
	ErrorCountQuery QueryType = "error_count"
	// RequestCountQuery is a query that should return a count of requests
	RequestCountQuery = "request_count"
	// ErrorRateQuery is a query that should return a rate of errors
	ErrorRateQuery = "error_rate"
	// ThresholdQuery is a query that should return a count to be thresholded
	ThresholdQuery = "threshold_query"
	// Percentile99 is a query that should return the 99th percentile latency
	Percentile99 = "percentile_99"
	// Percentile90 is a query that should return the 90th percentile latency
	Percentile90 = "percentile_90"
	// Percentile50 is a query that should return the 50th percentile latency
	Percentile50 = "percentile_50"
)

// Scan is necessary to retrieve a typed string QueryType from the db
func (u *QueryType) Scan(value interface{}) error { *u = QueryType(value.([]byte)); return nil }

// Value is necessary to store a typed string QueryType in the db
func (u QueryType) Value() (driver.Value, error) { return string(u), nil }

// AggregateType defines how points in a 5 minute bucket should be summarized.
type AggregateType string

const (
	// Sum means the points will be summed
	Sum AggregateType = "sum"
	// Average means the points will be averaged
	Average = "average"
	// Max means the points will be averaged
	Max = "max"
)

// Scan is necessary to retrieve a typed string AggregateType from the db
func (u *AggregateType) Scan(value interface{}) error { *u = AggregateType(value.([]byte)); return nil }

// Value is necessary to store a typed string AggregateType in the db
func (u AggregateType) Value() (driver.Value, error) { return string(u), nil }

// Query defines the
type Query struct {
	gorm.Model
	QueryType QueryType
	MetricID  uint
	// DataSource DataSource
	Query         string `gorm:"type:varchar(1000)"`
	AggregateType AggregateType
}

// GetQueryStringOfType returns the query string from a list of queries with a specific type
func GetQueryStringOfType(queries []*Query, typ QueryType) string {
	for _, elm := range queries {
		if elm.QueryType == typ {
			return elm.Query
		}
	}
	return ""
}

// GetQueryOfType returns the query from a list of queries with a specific type
func GetQueryOfType(queries []*Query, typ QueryType) *Query {
	for _, elm := range queries {
		if elm.QueryType == typ {
			return elm
		}
	}
	return nil
}
