package catalog

import (
	"errors"
	"fmt"

	"github.com/jinzhu/gorm"
)

// Service Defines the schema for the services table and contains the information about a service
type Service struct {
	gorm.Model
	Name                  string `gorm:"not null;unique"` // Globally unique human service name
	Description           string `gorm:"size:2000"`
	TeamID                uint   // Service owner ID
	Known                 bool
	ServiceTypeID         uint
	ServiceType           *ServiceType
	Environment           string
	Components            []*Component
	Team                  *Team
	AvailabilityObjective float64 `sql:"DEFAULT:99.9"`
	ServiceAudits         []*ServiceAudit
}

// ServiceType provides useful metadata surrounding service types for the
// user interface.
// This should be deprecated soon
type ServiceType struct {
	gorm.Model
	Name        string
	Label       string
	Description string `gorm:"size:2000"`
}

// GetServiceTypeMap returns a map from internal service type
// labels to the actual struct from the table.
// This is an intermediate use case until we can kill the table off
func GetServiceTypeMap() (map[string]*ServiceType, error) {
	serviceTypes := make(map[string]*ServiceType)
	sts, err := catalogDB.GetServiceTypes()
	if err != nil {
		return nil, err
	}
	for _, serviceType := range sts {
		serviceTypes[serviceType.Label] = serviceType
	}
	return serviceTypes, nil
}

// URL returns the endpoint for the show page for a specific service
func (s *Service) URL() string {
	return "/services/" + fmt.Sprint(s.ID)
}

// Returns only the latest service audit of each type
func (s *Service) LatestServiceAudits() []*ServiceAudit {
	audits := make(map[string]*ServiceAudit)
	for _, sa := range s.ServiceAudits {
		// If the audit type hasnt been seen yet, add this audit right away
		if _, found := audits[sa.AuditType]; !found {
			audits[sa.AuditType] = sa
		} else {
			// audit type already in the map, so only use this one if it's newer
			if sa.AuditTime.After(audits[sa.AuditType].AuditTime) {
				audits[sa.AuditType] = sa
			}
		}
	}
	// Turn the map back into an array
	res := make([]*ServiceAudit, 0)
	for _, v := range audits {
		res = append(res, v)
	}
	return res
}

// DefaultService returns a zero service
func DefaultService() *Service {
	// Most of these wouldn't be different
	// if they were just excluded (zero values in Go),
	// but let's be explicit
	return &Service{
		Name:        "",
		Description: "",
		Environment: "",
		Known:       false,
	}
}

// BeforeSave gorm-invoked callback that is used for validation
func (s *Service) BeforeSave() error {
	if s.Name == "" {
		return errors.New("service name cannot be empty")
	}
	return nil
}
