package catalog

import (
	"errors"
	"fmt"
	"time"

	"github.com/jinzhu/gorm"
)

// ServiceAuditTypes maintains a list of valid audit types
// found in the service_audits table
var ServiceAuditTypes = []string{
	"availability",
	"component",
	"latency",
	"owner",
}

var ServiceAuditActions = []string{
	"validated",
	"invalidated",
	"expired",
}

// ServiceAudit Defines the schema for the service_audits table
type ServiceAudit struct {
	gorm.Model
	ServiceID uint
	AuditType string
	Auditor   string
	Action    string
	AuditTime time.Time
}

// URL returns the endpoint for the show page for a specific service
func (sa *ServiceAudit) URL() string {
	return fmt.Sprintf("/services/%d/audits", sa.ID)
}

// BeforeSave is a gorm-invoked callback that is used for validation
func (sa *ServiceAudit) BeforeSave() error {
	if sa.ServiceID == 0 {
		return errors.New("service_id cannot be empty")
	}
	if sa.Auditor == "" {
		return errors.New("auditor cannot be empty")
	}
	if !sa.validAuditType() {
		return errors.New("audit_type cannot be omitted or invalid")
	}
	if !sa.validAuditAction() {
		return errors.New("action cannot be omitted or invalid")
	}
	return nil
}

func (sa *ServiceAudit) validAuditType() bool {
	for _, validType := range ServiceAuditTypes {
		if sa.AuditType == validType {
			return true
		}
	}
	return false
}

func (sa *ServiceAudit) validAuditAction() bool {
	for _, validAction := range ServiceAuditActions {
		if sa.Action == validAction {
			return true
		}
	}
	return false
}
