package manager

import (
	"context"
	"io/ioutil"
	"time"

	"code.justin.tv/systems/sandstorm/inventory/heartbeat"
	"code.justin.tv/systems/sandstorm/queue"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sts"
	uuid "github.com/satori/go.uuid"
	"github.com/sirupsen/logrus"
)

// API describes the manager interface
type API interface {
	AuditTableName() string
	CheckKMS() error
	CheckTable() error
	Copy(source, destination string) error
	CreateTable() error
	Decrypt(secret *Secret) error
	Delete(secretName string) error
	Exist(secretName string) (bool, error)
	CrossEnvironmentSecretsSet(secretNames []string) (crossEnvSecrets map[string]struct{}, err error)
	FlushHeartbeat(ctx context.Context)
	Get(secretName string) (*Secret, error)
	GetEncrypted(secretName string) (*Secret, error)
	GetVersion(secretName string, version int64) (*Secret, error)
	GetVersionsEncrypted(secretName string, limit int64, offsetKey int64) (*VersionsPage, error)
	List() ([]*Secret, error)
	ListenForUpdates() error
	ListNamespace(namespace string) ([]*Secret, error)
	ListNamespaces() ([]string, error)
	NamespaceTableName() string
	Patch(patchInput *PatchInput) error
	Post(secret *Secret) error
	Put(secret *Secret) error
	Revert(name string, version int64) error
	RegisterSecretUpdateCallback(secretName string, callback func(secretName string)) error
	Seal(secret *Secret) error
	StopListeningForUpdates() error
	TableName() string
	CleanUp() error
}

const version = "1.0"

const (
	dynamoDBKeyTombstone        = "tombstone"
	dynamoDBKeyActionUser       = "action_user"
	dynamoDBKeyChangelogEnabled = "changelog_enabled"
	dynamoDBKeyServiceName      = "service_name"
)

// Manager is the struct that a caller interacts with for managing
// secrets.
type Manager struct {
	Envelope                Enveloper
	DynamoDB                dynamodbiface.DynamoDBAPI
	Config                  Config
	Logger                  logrus.FieldLogger
	inventoryClient         heartbeat.API
	queue                   queue.Queuer
	cache                   *cache
	stopListeningForUpdates chan struct{}
	secretCallbacks         map[string][]func(string)
}

// Config is a struct for configuring the manager
// XXX need to be able to provide an aws.Config for STS credentials,
// region et c
type Config struct {
	// AWS config structure. If left nil, will be replaced by a
	// default config with region us-west-2.
	AWSConfig *aws.Config
	// KMS KeyId - this is either a key alias prefixed by 'alias/',
	// a key ARN or a key UUID
	KeyID string
	// DynamoDB Table name
	TableName string
	// Provisioned Read and Write throughput when creating the
	// table, not used in normal operations. If left to 0 these
	// will be replaced by 1 by default.
	ProvisionedThroughputRead  int64
	ProvisionedThroughputWrite int64
	StatsdHostPort             string
	StatsdPrefix               string
	Logger                     logrus.FieldLogger

	Host               string
	InventoryRoleARN   string
	InventoryStatusURL string
	InventoryInterval  time.Duration
	ServiceName        string

	Queue      queue.Config
	InstanceID string

	//@TODO: Temporary once the clients are migrated, remove this
	LogToChangelog bool
	//User performing the action
	ActionUser string
}

var noopLogger = &logrus.Logger{
	Out: ioutil.Discard,
}

const unknownServiceName = "unknown-service"

// DefaultConfig returns a config struct with some sane defaults that
// is merged with the provided config in New
func DefaultConfig() Config {
	return Config{
		ProvisionedThroughputRead:  1,
		ProvisionedThroughputWrite: 1,
		AWSConfig:                  &aws.Config{Region: aws.String("us-west-2")},
		Logger:                     noopLogger,
		InventoryRoleARN:           "arn:aws:iam::854594403332:role/inventory-gateway-execute-api-invoke-production",
		KeyID:                      "alias/sandstorm-production",
		TableName:                  "sandstorm-production",
	}
}

func mergeConfig(provided *Config, defConfig Config) {
	if provided.AWSConfig == nil {
		provided.AWSConfig = defConfig.AWSConfig
	}
	if provided.ProvisionedThroughputRead == 0 {
		provided.ProvisionedThroughputRead = defConfig.ProvisionedThroughputRead
	}
	if provided.ProvisionedThroughputWrite == 0 {
		provided.ProvisionedThroughputWrite = defConfig.ProvisionedThroughputWrite
	}
	if provided.Logger == nil {
		provided.Logger = defConfig.Logger
	}
	if provided.InventoryRoleARN == "" {
		provided.InventoryRoleARN = defConfig.InventoryRoleARN
	}
	if provided.InstanceID == "" {
		provided.InstanceID = uuid.NewV4().String()
	}
	if provided.KeyID == "" {
		provided.KeyID = defConfig.KeyID
	}
	if provided.TableName == "" {
		provided.TableName = defConfig.TableName
	}
	if provided.ActionUser == "" {
		provided.ActionUser = getCallerIdentity(provided.AWSConfig)
	}
}

// New creates a Manager from config. Merges configuration with
// DefaultConfig()
func New(config Config) *Manager {
	defConfig := DefaultConfig()
	mergeConfig(&config, defConfig)

	session := session.New(config.AWSConfig)

	ddb := dynamodb.New(session)

	mgr := &Manager{
		DynamoDB: ddb,
		Config:   config,
		Envelope: &Envelope{
			KMS: kms.New(session),
		},
		Logger: config.Logger,
		cache:  &cache{},
		queue:  queue.New(sqs.New(session), sns.New(session), config.Queue, config.Logger),
		stopListeningForUpdates: make(chan struct{}),
		secretCallbacks:         make(map[string][]func(secretName string)),
	}

	mgr.inventoryClient = heartbeat.New(
		stscreds.NewCredentials(session, config.InventoryRoleARN),
		&heartbeat.Config{
			Interval:       config.InventoryInterval,
			URL:            config.InventoryStatusURL,
			Service:        config.ServiceName,
			Region:         *config.AWSConfig.Region,
			Host:           config.Host,
			InstanceID:     config.InstanceID,
			ManagerVersion: version,
		},
		config.Logger,
	)
	go mgr.inventoryClient.Start()

	return mgr
}

func (mgr *Manager) newCache() {
	mgr.cache = &cache{
		secretMap: make(map[string]*Secret),
	}
}

// ListenForUpdates manager listens on the sandstorm queue for secret updates.
// It also configures the manager to cache secret plaintext in memory. When an
// update event is received for a secret, that secret is expunged from the in
// memory cache, so the next Get call will fetch that secret from DynamoDB.
func (mgr *Manager) ListenForUpdates() (err error) {
	mgr.newCache()
	err = mgr.queue.Setup()
	if err != nil {
		return
	}

	go func() {
		for {
			select {
			case <-mgr.stopListeningForUpdates:
				return
			default:
				err := mgr.listenForSecretUpdates()
				if err != nil {
					mgr.Logger.Errorf("Failed to listen for secret update. err: %s", err.Error())
				}
			}
		}
	}()
	return
}

func (mgr *Manager) listenForSecretUpdates() (err error) {
	errChan := make(chan error)
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	go func() {
		queueSecret, err := mgr.queue.PollForSecret(ctx)
		if err != nil {
			errChan <- err
			return
		}
		if queueSecret != nil {
			secretName := queueSecret.Name.S
			mgr.cache.Delete(secretName)
			mgr.triggerSecretUpdateCallbacks(secretName)
		}
		errChan <- nil
		return
	}()

	select {
	case <-mgr.stopListeningForUpdates:
		err = <-errChan
	case err = <-errChan:
	}
	return
}

// StopListeningForUpdates manager stops listening to secret update queue and
// deletes the queue
func (mgr *Manager) StopListeningForUpdates() (err error) {
	mgr.cache = &cache{}
	close(mgr.stopListeningForUpdates)
	return mgr.queue.Delete()
}

// CleanUp cleans exits any go routines spawned
func (mgr *Manager) CleanUp() (err error) {
	inventoryStopErr := mgr.inventoryClient.Stop()
	if inventoryStopErr != nil {
		mgr.Logger.Debugf("error stopping inventory client: %s", inventoryStopErr.Error())
	}
	return
}

// FlushHeartbeat sends a signal to immediately report heartbeat
func (mgr *Manager) FlushHeartbeat(ctx context.Context) {
	mgr.inventoryClient.FlushHeartbeat(ctx)
}

//return arn of the caller identity, can be role or user
func getCallerIdentity(awsConfig *aws.Config) (callerArn string) {
	svc := sts.New(session.New(awsConfig))
	input := &sts.GetCallerIdentityInput{}

	result, err := svc.GetCallerIdentity(input)
	if err != nil {
		//caller will be set to "" if we cannot get caller identity.
		return
	}
	callerArn = aws.StringValue(result.Arn)
	return
}
